package topology

import (
	"reflect"
	"testing"

	eventservice "code.fbi.h-da.de/danet/gosdn/controller/eventService"
	eventInterfaces "code.fbi.h-da.de/danet/gosdn/controller/interfaces/event"
	query "code.fbi.h-da.de/danet/gosdn/controller/store"
	"code.fbi.h-da.de/danet/gosdn/controller/topology/links"
	"code.fbi.h-da.de/danet/gosdn/controller/topology/nodes"
	"code.fbi.h-da.de/danet/gosdn/controller/topology/ports"
	"code.fbi.h-da.de/danet/gosdn/controller/topology/store"

	"github.com/google/uuid"
)

func getTestSourceNode() nodes.Node {
	return nodes.Node{
		ID:   uuid.MustParse("44fb4aa4-c53c-4cf9-a081-5aabc61c7610"),
		Name: "Test-Source-Node",
	}
}

func getTestTargetNode() nodes.Node {
	return nodes.Node{
		ID:   uuid.MustParse("44fb4aa4-c53c-4cf9-a081-5aabc61c7612"),
		Name: "Test-Target-Node",
	}
}

func getTestSourcePort() ports.Port {
	return ports.Port{
		ID:   uuid.MustParse("1fa479e7-d393-4d45-822d-485cc1f05fce"),
		Name: "Test-Source-Port",
	}
}

func getTestTargetPort() ports.Port {
	return ports.Port{
		ID:   uuid.MustParse("1fa479e7-d393-4d45-822d-485cc1f05fc2"),
		Name: "Test-Target-Port",
	}
}

func getTestLink() links.Link {
	return links.Link{
		ID:         uuid.MustParse("5eb474f1-428e-4503-ba68-dcf9bef53467"),
		Name:       "Test-Link",
		SourceNode: getTestSourceNode(),
		TargetNode: getTestTargetNode(),
		SourcePort: getTestSourcePort(),
		TargetPort: getTestTargetPort(),
	}
}

func getEmptyLink() links.Link {
	return links.Link{}
}

func getTestStoreWithLinks(t *testing.T, nodes []links.Link) Store {
	store := store.NewGenericStore[links.Link]()

	for _, node := range nodes {
		err := store.Add(node)
		if err != nil {
			t.Fatalf("failed to prepare test store while adding node: %v", err)
		}
	}

	return store
}

func getTestStoreWithNodes(t *testing.T, nodesToAdd []nodes.Node) nodes.Store {
	store := store.NewGenericStore[nodes.Node]()

	for _, node := range nodesToAdd {
		err := store.Add(node)
		if err != nil {
			t.Fatalf("failed to prepare test store while adding node: %v", err)
		}
	}

	return store
}

func getTestStoreWithPorts(t *testing.T, portsToAdd []ports.Port) ports.Store {
	store := store.NewGenericStore[ports.Port]()

	for _, port := range portsToAdd {
		err := store.Add(port)
		if err != nil {
			t.Fatalf("failed to prepare test store while adding port: %v", err)
		}
	}

	return store
}

func TestNewTopologyService(t *testing.T) {
	type args struct {
		store        Store
		nodeService  nodes.Service
		portService  ports.Service
		eventService eventInterfaces.Service
	}
	tests := []struct {
		name string
		args args
		want Service
	}{
		{
			name: "should create a new topology service",
			args: args{
				store:        getTestStoreWithLinks(t, []links.Link{}),
				nodeService:  nodes.NewNodeService(getTestStoreWithNodes(t, []nodes.Node{}), eventservice.NewMockEventService()),
				portService:  ports.NewPortService(getTestStoreWithPorts(t, []ports.Port{}), eventservice.NewMockEventService()),
				eventService: eventservice.NewMockEventService(),
			},
			want: NewTopologyService(
				getTestStoreWithLinks(t, []links.Link{}),
				nodes.NewNodeService(getTestStoreWithNodes(t, []nodes.Node{}), eventservice.NewMockEventService()),
				ports.NewPortService(getTestStoreWithPorts(t, []ports.Port{}), eventservice.NewMockEventService()),
				eventservice.NewMockEventService(),
			),
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			if got := NewTopologyService(
				tt.args.store,
				tt.args.nodeService,
				tt.args.portService,
				tt.args.eventService,
			); !reflect.DeepEqual(got, tt.want) {
				t.Errorf("NewNodeService() = %v, want %v", got, tt.want)
			}
		})
	}
}

func TestTopologyService_AddLink(t *testing.T) {
	type fields struct {
		store        Store
		nodeService  nodes.Service
		portService  ports.Service
		eventService eventInterfaces.Service
	}
	type args struct {
		link links.Link
	}
	tests := []struct {
		name    string
		fields  fields
		args    args
		want    links.Link
		wantErr bool
	}{
		{
			name: "should add a link to the store",
			fields: fields{
				store: store.NewGenericStore[links.Link](),
				nodeService: nodes.NewNodeService(
					getTestStoreWithNodes(
						t,
						[]nodes.Node{getTestSourceNode(), getTestTargetNode()},
					),
					eventservice.NewMockEventService(),
				),
				portService: ports.NewPortService(
					getTestStoreWithPorts(
						t,
						[]ports.Port{getTestSourcePort(), getTestTargetPort()},
					),
					eventservice.NewMockEventService(),
				),
				eventService: eventservice.NewMockEventService(),
			},
			args: args{
				link: getTestLink(),
			},
			want:    getTestLink(),
			wantErr: false,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			p := &TopoService{
				store:        tt.fields.store,
				nodeService:  tt.fields.nodeService,
				portService:  tt.fields.portService,
				eventService: tt.fields.eventService,
			}
			err := p.AddLink(tt.args.link)
			if (err != nil) != tt.wantErr {
				t.Errorf("service.AddLink() error = %v, wantErr %v", err, tt.wantErr)
				return
			}

			got, err := p.Get(query.Query{ID: tt.args.link.ID})
			if (err != nil) != tt.wantErr {
				t.Errorf("service.Get() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if !reflect.DeepEqual(got, tt.want) {
				t.Errorf("service.Get() = %v, want %v", got, tt.want)
			}
		})
	}
}

func TestTopologyService_Update(t *testing.T) {
	type fields struct {
		store        Store
		nodeService  nodes.Service
		portService  ports.Service
		eventService eventInterfaces.Service
	}
	type args struct {
		link links.Link
	}
	tests := []struct {
		name    string
		fields  fields
		args    args
		want    links.Link
		wantErr bool
	}{
		{
			name: "should update an existing link",
			fields: fields{
				store:        getTestStoreWithLinks(t, []links.Link{getTestLink()}),
				nodeService:  nodes.NewNodeService(getTestStoreWithNodes(t, []nodes.Node{}), eventservice.NewMockEventService()),
				portService:  ports.NewPortService(getTestStoreWithPorts(t, []ports.Port{}), eventservice.NewMockEventService()),
				eventService: eventservice.NewMockEventService(),
			},
			args: args{
				link: getTestLink(),
			},
			want:    getTestLink(),
			wantErr: false,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			p := &TopoService{
				store:        tt.fields.store,
				nodeService:  tt.fields.nodeService,
				portService:  tt.fields.portService,
				eventService: tt.fields.eventService,
			}
			err := p.UpdateLink(tt.args.link)
			if (err != nil) != tt.wantErr {
				t.Errorf("service.Update() error = %v, wantErr %v", err, tt.wantErr)
				return
			}

			got, err := p.Get(query.Query{ID: tt.args.link.ID})
			if (err != nil) != tt.wantErr {
				t.Errorf("service.Get() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if !reflect.DeepEqual(got, tt.want) {
				t.Errorf("service.Get() = %v, want %v", got, tt.want)
			}
		})
	}
}

func TestNodeService_Delete(t *testing.T) {
	type fields struct {
		store        Store
		nodeService  nodes.Service
		portService  ports.Service
		eventService eventInterfaces.Service
	}
	type args struct {
		link links.Link
	}
	tests := []struct {
		name    string
		fields  fields
		args    args
		want    links.Link
		wantErr bool
	}{
		{
			name: "should delete an existing link",
			fields: fields{
				store:        getTestStoreWithLinks(t, []links.Link{getTestLink()}),
				nodeService:  nodes.NewNodeService(getTestStoreWithNodes(t, []nodes.Node{}), eventservice.NewMockEventService()),
				portService:  ports.NewPortService(getTestStoreWithPorts(t, []ports.Port{}), eventservice.NewMockEventService()),
				eventService: eventservice.NewMockEventService(),
			},
			args: args{
				link: getTestLink(),
			},
			wantErr: false,
		},
		{
			name: "should fail if a node does not exists",
			fields: fields{
				store:       store.NewGenericStore[links.Link](),
				nodeService: nodes.NewNodeService(getTestStoreWithNodes(t, []nodes.Node{}), eventservice.NewMockEventService()),
				portService: ports.NewPortService(getTestStoreWithPorts(t, []ports.Port{}), eventservice.NewMockEventService()),
			},
			args: args{
				link: getTestLink(),
			},
			wantErr: true,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			p := &TopoService{
				store:        tt.fields.store,
				nodeService:  tt.fields.nodeService,
				portService:  tt.fields.portService,
				eventService: tt.fields.eventService,
			}
			if err := p.DeleteLink(tt.args.link); (err != nil) != tt.wantErr {
				t.Errorf("service.Delete() error = %v, wantErr %v", err, tt.wantErr)
			}
		})
	}
}

func TestTopologyService_Get(t *testing.T) {
	type fields struct {
		store        Store
		nodeService  nodes.Service
		portService  ports.Service
		eventService eventInterfaces.Service
	}
	type args struct {
		query query.Query
	}
	tests := []struct {
		name    string
		fields  fields
		args    args
		want    links.Link
		wantErr bool
	}{
		{
			name: "should error if link with uuid is not in store",
			fields: fields{
				store:       getTestStoreWithLinks(t, []links.Link{}),
				nodeService: nodes.NewNodeService(getTestStoreWithNodes(t, []nodes.Node{}), eventservice.NewMockEventService()),
				portService: ports.NewPortService(getTestStoreWithPorts(t, []ports.Port{}), eventservice.NewMockEventService()),
			},
			args: args{
				query: query.Query{
					ID:   getTestLink().ID,
					Name: getTestLink().Name,
				},
			},
			want:    getEmptyLink(),
			wantErr: true,
		},
		{
			name: "should return link that is in the store",
			fields: fields{
				store: getTestStoreWithLinks(t, []links.Link{getTestLink()}),
			},
			args: args{
				query: query.Query{
					ID:   getTestLink().ID,
					Name: getTestLink().Name,
				},
			},
			want:    getTestLink(),
			wantErr: false,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			p := &TopoService{
				store:        tt.fields.store,
				nodeService:  tt.fields.nodeService,
				portService:  tt.fields.portService,
				eventService: tt.fields.eventService,
			}
			got, err := p.Get(tt.args.query)
			if (err != nil) != tt.wantErr {
				t.Errorf("service.Get() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if !reflect.DeepEqual(got, tt.want) {
				t.Errorf("service.Get() = %v, want %v", got, tt.want)
			}
		})
	}
}

func TestTopologyService_GetAll(t *testing.T) {
	type fields struct {
		store        Store
		nodeService  nodes.Service
		portService  ports.Service
		eventService eventInterfaces.Service
	}
	tests := []struct {
		name    string
		fields  fields
		want    []links.Link
		wantErr bool
	}{
		{
			name: "should get all stored links",
			fields: fields{
				store:        getTestStoreWithLinks(t, []links.Link{getTestLink()}),
				nodeService:  nodes.NewNodeService(getTestStoreWithNodes(t, []nodes.Node{}), eventservice.NewMockEventService()),
				portService:  ports.NewPortService(getTestStoreWithPorts(t, []ports.Port{}), eventservice.NewMockEventService()),
				eventService: eventservice.NewMockEventService(),
			},
			want:    []links.Link{getTestLink()},
			wantErr: false,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			p := &TopoService{
				store:        tt.fields.store,
				nodeService:  tt.fields.nodeService,
				portService:  tt.fields.portService,
				eventService: tt.fields.eventService,
			}
			got, err := p.GetAll()
			if (err != nil) != tt.wantErr {
				t.Errorf("service.GetAll() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if !reflect.DeepEqual(got, tt.want) {
				t.Errorf("service.GetAll() = %v, want %v", got, tt.want)
			}
		})
	}
}
