package routingtables

import (
	"fmt"

	"code.fbi.h-da.de/danet/gosdn/controller/customerrs"
	"code.fbi.h-da.de/danet/gosdn/controller/nucleus/database"
	query "code.fbi.h-da.de/danet/gosdn/controller/store"

	"github.com/google/uuid"
	"go.mongodb.org/mongo-driver/bson"
	"go.mongodb.org/mongo-driver/bson/primitive"
	"go.mongodb.org/mongo-driver/mongo/options"
)

// Store defines a RoutingTable store interface.
type Store interface {
	Add(RoutingTable) error
	Update(RoutingTable) error
	Delete(RoutingTable) error
	Get(query.Query) (RoutingTable, error)
	GetAll() ([]RoutingTable, error)
}

// DatabaseRoutingTableStore is a database store for routingTables.
type DatabaseRoutingTableStore struct {
	storeName string
}

// NewDatabaseRoutingTableStore returns a RoutingTableStore.
func NewDatabaseRoutingTableStore() Store {
	return &DatabaseRoutingTableStore{
		storeName: fmt.Sprint("routing-table-store.json"),
	}
}

// Get takes a routing-tables's UUID or name and returns the entries.
func (s *DatabaseRoutingTableStore) Get(query query.Query) (RoutingTable, error) {
	var loadedRoutingTable RoutingTable

	if query.ID.String() != "" {
		loadedRoutingTable, err := s.getByID(query.ID)
		if err != nil {
			return loadedRoutingTable, customerrs.CouldNotFindError{ID: query.ID, Name: query.Name}
		}

		return loadedRoutingTable, nil
	}

	loadedRoutingTable, err := s.getByName(query.Name)
	if err != nil {
		return loadedRoutingTable, customerrs.CouldNotFindError{ID: query.ID, Name: query.Name}
	}

	return loadedRoutingTable, nil
}

func (s *DatabaseRoutingTableStore) getByID(idOfRoutingTable uuid.UUID) (routingTable RoutingTable, err error) {
	client, ctx, cancel := database.GetMongoConnection()
	defer cancel()
	defer func() {
		if ferr := client.Disconnect(ctx); ferr != nil {
			fErrString := ferr.Error()
			err = fmt.Errorf("InternalError=%w DeferError=%+s", err, fErrString)
		}
	}()

	db := client.Database(database.DatabaseName)
	collection := db.Collection(s.storeName)
	result := collection.FindOne(ctx, bson.D{primitive.E{Key: "_id", Value: idOfRoutingTable.String()}})
	if result == nil {
		return routingTable, customerrs.CouldNotFindError{ID: idOfRoutingTable}
	}

	err = result.Decode(&routingTable)
	if err != nil {
		return routingTable, customerrs.CouldNotMarshallError{Identifier: idOfRoutingTable, Type: routingTable, Err: err}
	}

	return routingTable, nil
}

func (s *DatabaseRoutingTableStore) getByName(nameOfRoutingTable string) (loadedRoutingTable RoutingTable, err error) {
	client, ctx, cancel := database.GetMongoConnection()
	defer cancel()
	defer func() {
		if ferr := client.Disconnect(ctx); ferr != nil {
			fErrString := ferr.Error()
			err = fmt.Errorf("InternalError=%w DeferError=%+s", err, fErrString)
		}
	}()

	db := client.Database(database.DatabaseName)
	collection := db.Collection(s.storeName)
	result := collection.FindOne(ctx, bson.D{primitive.E{Key: "name", Value: nameOfRoutingTable}})
	if result == nil {
		return loadedRoutingTable, customerrs.CouldNotFindError{Name: nameOfRoutingTable}
	}

	err = result.Decode(&loadedRoutingTable)
	if err != nil {
		return loadedRoutingTable, customerrs.CouldNotMarshallError{Type: loadedRoutingTable, Err: err}
	}

	return loadedRoutingTable, nil
}

// GetAll returns all stored routingTables.
func (s *DatabaseRoutingTableStore) GetAll() (loadedRoutingTable []RoutingTable, err error) {
	client, ctx, cancel := database.GetMongoConnection()
	defer cancel()
	defer func() {
		if ferr := client.Disconnect(ctx); ferr != nil {
			fErrString := ferr.Error()
			err = fmt.Errorf("InternalError=%w DeferError=%+s", err, fErrString)
		}
	}()

	db := client.Database(database.DatabaseName)
	collection := db.Collection(s.storeName)

	cursor, err := collection.Find(ctx, bson.D{})
	if err != nil {
		return []RoutingTable{}, err
	}
	defer func() {
		if ferr := cursor.Close(ctx); ferr != nil {
			fErrString := ferr.Error()
			err = fmt.Errorf("InternalError=%w DeferError=%+s", err, fErrString)
		}
	}()

	err = cursor.All(ctx, &loadedRoutingTable)
	if err != nil {
		return loadedRoutingTable, customerrs.CouldNotMarshallError{Type: loadedRoutingTable, Err: err}
	}

	return loadedRoutingTable, nil
}

// Add adds a RoutingTable to the store.
func (s *DatabaseRoutingTableStore) Add(routingTable RoutingTable) (err error) {
	client, ctx, cancel := database.GetMongoConnection()
	defer cancel()
	defer func() {
		if ferr := client.Disconnect(ctx); ferr != nil {
			fErrString := ferr.Error()
			err = fmt.Errorf("InternalError=%w DeferError=%+s", err, fErrString)
		}
	}()

	_, err = client.Database(database.DatabaseName).
		Collection(s.storeName).
		InsertOne(ctx, routingTable)
	if err != nil {
		return customerrs.CouldNotCreateError{Identifier: routingTable.ID, Type: routingTable, Err: err}
	}

	return nil
}

// Update updates a existing routingTable.
func (s *DatabaseRoutingTableStore) Update(routingTable RoutingTable) (err error) {
	var updatedLoadedRoutingTable RoutingTable

	client, ctx, cancel := database.GetMongoConnection()
	defer cancel()
	defer func() {
		if ferr := client.Disconnect(ctx); ferr != nil {
			fErrString := ferr.Error()
			err = fmt.Errorf("InternalError=%w DeferError=%+s", err, fErrString)
		}
	}()

	update := bson.D{primitive.E{Key: "$set", Value: routingTable}}

	upsert := false
	after := options.After
	opt := options.FindOneAndUpdateOptions{
		Upsert:         &upsert,
		ReturnDocument: &after,
	}

	err = client.Database(database.DatabaseName).
		Collection(s.storeName).
		FindOneAndUpdate(
			ctx, bson.M{"_id": routingTable.ID.String()}, update, &opt).
		Decode(&updatedLoadedRoutingTable)
	if err != nil {
		return customerrs.CouldNotUpdateError{Identifier: routingTable.ID, Type: routingTable, Err: err}
	}

	return nil
}

// Delete deletes a node from the node store.
func (s *DatabaseRoutingTableStore) Delete(routingTable RoutingTable) (err error) {
	client, ctx, cancel := database.GetMongoConnection()
	defer cancel()
	defer func() {
		if ferr := client.Disconnect(ctx); ferr != nil {
			fErrString := ferr.Error()
			err = fmt.Errorf("InternalError=%w DeferError=%+s", err, fErrString)
		}
	}()

	db := client.Database(database.DatabaseName)
	collection := db.Collection(s.storeName)
	_, err = collection.DeleteOne(ctx, bson.D{primitive.E{Key: routingTable.ID.String()}})
	if err != nil {
		return err
	}

	return nil
}
