package routingtables

import (
	"reflect"
	"testing"

	eventservice "code.fbi.h-da.de/danet/gosdn/controller/eventService"
	eventInterfaces "code.fbi.h-da.de/danet/gosdn/controller/interfaces/event"
	query "code.fbi.h-da.de/danet/gosdn/controller/store"
	"code.fbi.h-da.de/danet/gosdn/controller/topology/nodes"
	"code.fbi.h-da.de/danet/gosdn/controller/topology/ports"
	"code.fbi.h-da.de/danet/gosdn/controller/topology/store"

	"github.com/google/uuid"
)

func getTestNode() nodes.Node {
	return nodes.Node{
		ID:   uuid.MustParse("44fb4aa4-c53c-4cf9-a081-5aabc61c7610"),
		Name: "Test-Source-Node",
	}
}

func getTestPort() ports.Port {
	return ports.Port{
		ID:   uuid.MustParse("1fa479e7-d393-4d45-822d-485cc1f05fc2"),
		Name: "Test-Target-Port",
	}
}

func getTestRoute() Route {
	return Route{
		ID:            uuid.MustParse("1fa479e7-d393-4d45-822d-485cc1f05f12"),
		TargetIPRange: "10.13.37.0/24",
		PortID:        uuid.MustParse("1fa479e7-d393-4d45-822d-485cc1f05fc2"),
		Metric:        "1",
	}
}

func getTestRoutingTable() RoutingTable {
	return RoutingTable{
		ID:     uuid.MustParse("1fa479e7-d393-4d45-822d-485cc1f05f34"),
		NodeID: uuid.MustParse("44fb4aa4-c53c-4cf9-a081-5aabc61c7610"),
		Routes: []Route{getTestRoute()},
	}
}

func getTestStoreWithRoutingTables(t *testing.T, routingTables []RoutingTable) Store {
	store := store.NewGenericStore[RoutingTable]()

	for _, rt := range routingTables {
		err := store.Add(rt)
		if err != nil {
			t.Fatalf("failed to prepare test store while adding routing table: %v", err)
		}
	}

	return store
}

func getTestStoreWithNodes(t *testing.T, nodesToAdd []nodes.Node) nodes.Store {
	store := store.NewGenericStore[nodes.Node]()

	for _, node := range nodesToAdd {
		err := store.Add(node)
		if err != nil {
			t.Fatalf("failed to prepare test store while adding node: %v", err)
		}
	}

	return store
}

func getTestStoreWithPorts(t *testing.T, portsToAdd []ports.Port) ports.Store {
	store := store.NewGenericStore[ports.Port]()

	for _, port := range portsToAdd {
		err := store.Add(port)
		if err != nil {
			t.Fatalf("failed to prepare test store while adding port: %v", err)
		}
	}

	return store
}

func TestNewRoutingTableService(t *testing.T) {
	type args struct {
		store        Store
		nodeService  nodes.Service
		portService  ports.Service
		eventService eventInterfaces.Service
	}
	tests := []struct {
		name string
		args args
		want Service
	}{
		{
			name: "should create a new topology service",
			args: args{
				store:        getTestStoreWithRoutingTables(t, []RoutingTable{}),
				nodeService:  nodes.NewNodeService(getTestStoreWithNodes(t, []nodes.Node{}), eventservice.NewMockEventService()),
				portService:  ports.NewPortService(getTestStoreWithPorts(t, []ports.Port{}), eventservice.NewMockEventService()),
				eventService: eventservice.NewMockEventService(),
			},
			want: NewRoutingTableService(
				getTestStoreWithRoutingTables(t, []RoutingTable{}),
				nodes.NewNodeService(getTestStoreWithNodes(t, []nodes.Node{}), eventservice.NewMockEventService()),
				ports.NewPortService(getTestStoreWithPorts(t, []ports.Port{}), eventservice.NewMockEventService()),
				eventservice.NewMockEventService(),
			),
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			if got := NewRoutingTableService(tt.args.store, tt.args.nodeService, tt.args.portService, tt.args.eventService); !reflect.DeepEqual(got, tt.want) {
				t.Errorf("NewNodeService() = %v, want %v", got, tt.want)
			}
		})
	}
}

func TestTopologyService_EnsureExists(t *testing.T) {
	type fields struct {
		store        Store
		nodeService  nodes.Service
		portService  ports.Service
		eventService eventInterfaces.Service
	}
	type args struct {
		routingTable RoutingTable
	}
	tests := []struct {
		name    string
		fields  fields
		args    args
		want    RoutingTable
		wantErr bool
	}{
		{
			name: "should add if a routing table that is not in the store",
			fields: fields{
				store: getTestStoreWithRoutingTables(t, []RoutingTable{}),
				nodeService: nodes.NewNodeService(
					getTestStoreWithNodes(
						t,
						[]nodes.Node{},
					),
					eventservice.NewMockEventService(),
				),
				portService: ports.NewPortService(
					getTestStoreWithPorts(
						t,
						[]ports.Port{},
					),
					eventservice.NewMockEventService(),
				),
				eventService: eventservice.NewMockEventService(),
			},
			args: args{
				routingTable: getTestRoutingTable(),
			},
			want:    getTestRoutingTable(),
			wantErr: false,
		},
		{
			name: "should return routing table that is in the store",
			fields: fields{
				store: getTestStoreWithRoutingTables(t, []RoutingTable{getTestRoutingTable()}),
				nodeService: nodes.NewNodeService(
					getTestStoreWithNodes(
						t,
						[]nodes.Node{getTestNode()},
					),
					eventservice.NewMockEventService(),
				),
				portService: ports.NewPortService(
					getTestStoreWithPorts(
						t,
						[]ports.Port{getTestPort()},
					),
					eventservice.NewMockEventService(),
				),
				eventService: eventservice.NewMockEventService(),
			},
			args: args{
				routingTable: getTestRoutingTable(),
			},
			want:    getTestRoutingTable(),
			wantErr: false,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			p := &RoutingTableService{
				store:        tt.fields.store,
				nodeService:  tt.fields.nodeService,
				portService:  tt.fields.portService,
				eventService: tt.fields.eventService,
			}
			got, err := p.EnsureExists(tt.args.routingTable)
			if (err != nil) != tt.wantErr {
				t.Errorf("RoutingTableService.EnsureExists() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if !reflect.DeepEqual(got, tt.want) {
				t.Errorf("RoutingTableService.EnsureExists() = %v, want %v", got, tt.want)
			}
		})
	}
}

func TestRoutingTableService_Update(t *testing.T) {
	type fields struct {
		store        Store
		nodeService  nodes.Service
		portService  ports.Service
		eventService eventInterfaces.Service
	}
	type args struct {
		routingTable RoutingTable
	}
	tests := []struct {
		name    string
		fields  fields
		args    args
		want    RoutingTable
		wantErr bool
	}{
		{
			name: "should update an existing routing table",
			fields: fields{
				store:        getTestStoreWithRoutingTables(t, []RoutingTable{getTestRoutingTable()}),
				nodeService:  nodes.NewNodeService(getTestStoreWithNodes(t, []nodes.Node{}), eventservice.NewMockEventService()),
				portService:  ports.NewPortService(getTestStoreWithPorts(t, []ports.Port{}), eventservice.NewMockEventService()),
				eventService: eventservice.NewMockEventService(),
			},
			args: args{
				routingTable: getTestRoutingTable(),
			},
			want:    getTestRoutingTable(),
			wantErr: false,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			p := &RoutingTableService{
				store:        tt.fields.store,
				nodeService:  tt.fields.nodeService,
				portService:  tt.fields.portService,
				eventService: tt.fields.eventService,
			}
			if err := p.Update(tt.args.routingTable); (err != nil) != tt.wantErr {
				t.Errorf("RoutingTableService.Update() error = %v, wantErr %v", err, tt.wantErr)
			}
		})
	}
}

func TestRoutingTableService_Delete(t *testing.T) {
	type fields struct {
		store        Store
		nodeService  nodes.Service
		portService  ports.Service
		eventService eventInterfaces.Service
	}
	type args struct {
		routingTable RoutingTable
	}
	tests := []struct {
		name    string
		fields  fields
		args    args
		want    RoutingTable
		wantErr bool
	}{
		{
			name: "should delete an existing routing table",
			fields: fields{
				store:        getTestStoreWithRoutingTables(t, []RoutingTable{getTestRoutingTable()}),
				nodeService:  nodes.NewNodeService(getTestStoreWithNodes(t, []nodes.Node{}), eventservice.NewMockEventService()),
				portService:  ports.NewPortService(getTestStoreWithPorts(t, []ports.Port{}), eventservice.NewMockEventService()),
				eventService: eventservice.NewMockEventService(),
			},
			args: args{
				routingTable: getTestRoutingTable(),
			},
			wantErr: false,
		},
		{
			name: "should fail if a routing table does not exists",
			fields: fields{
				store:        store.NewGenericStore[RoutingTable](),
				nodeService:  nodes.NewNodeService(getTestStoreWithNodes(t, []nodes.Node{}), eventservice.NewMockEventService()),
				portService:  ports.NewPortService(getTestStoreWithPorts(t, []ports.Port{}), eventservice.NewMockEventService()),
				eventService: eventservice.NewMockEventService(),
			},
			args: args{
				routingTable: getTestRoutingTable(),
			},
			wantErr: true,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			p := &RoutingTableService{
				store:        tt.fields.store,
				nodeService:  tt.fields.nodeService,
				portService:  tt.fields.portService,
				eventService: tt.fields.eventService,
			}
			if err := p.Delete(tt.args.routingTable); (err != nil) != tt.wantErr {
				t.Errorf("RoutingTableService.Delete() error = %v, wantErr %v", err, tt.wantErr)
			}
		})
	}
}

func TestRoutingTableService_Get(t *testing.T) {
	type fields struct {
		store       Store
		nodeService nodes.Service
		portService ports.Service
	}
	type args struct {
		query query.Query
	}
	tests := []struct {
		name    string
		fields  fields
		args    args
		want    RoutingTable
		wantErr bool
	}{
		{
			name: "should error if routing table with uuid is not in store",
			fields: fields{
				store:       getTestStoreWithRoutingTables(t, []RoutingTable{}),
				nodeService: nodes.NewNodeService(getTestStoreWithNodes(t, []nodes.Node{}), eventservice.NewMockEventService()),
				portService: ports.NewPortService(getTestStoreWithPorts(t, []ports.Port{}), eventservice.NewMockEventService()),
			},
			args: args{
				query: query.Query{
					ID: getTestRoutingTable().ID,
				},
			},
			want:    RoutingTable{},
			wantErr: true,
		},
		{
			name: "should return routing table that is in the store",
			fields: fields{
				store: getTestStoreWithRoutingTables(t, []RoutingTable{getTestRoutingTable()}),
			},
			args: args{
				query: query.Query{
					ID: getTestRoutingTable().ID,
				},
			},
			want:    getTestRoutingTable(),
			wantErr: false,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			p := &RoutingTableService{
				store:       tt.fields.store,
				nodeService: tt.fields.nodeService,
				portService: tt.fields.portService,
			}
			got, err := p.Get(tt.args.query)
			if (err != nil) != tt.wantErr {
				t.Errorf("RoutingTableService.Get() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if !reflect.DeepEqual(got, tt.want) {
				t.Errorf("RoutingTableService.Get() = %v, want %v", got, tt.want)
			}
		})
	}
}

func TestRoutingTableService_GetAll(t *testing.T) {
	type fields struct {
		store       Store
		nodeService nodes.Service
		portService ports.Service
	}
	tests := []struct {
		name    string
		fields  fields
		want    []RoutingTable
		wantErr bool
	}{
		{
			name: "should get all stored routing tables",
			fields: fields{
				store:       getTestStoreWithRoutingTables(t, []RoutingTable{getTestRoutingTable()}),
				nodeService: nodes.NewNodeService(getTestStoreWithNodes(t, []nodes.Node{}), eventservice.NewMockEventService()),
				portService: ports.NewPortService(getTestStoreWithPorts(t, []ports.Port{}), eventservice.NewMockEventService()),
			},
			want:    []RoutingTable{getTestRoutingTable()},
			wantErr: false,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			p := &RoutingTableService{
				store:       tt.fields.store,
				nodeService: tt.fields.nodeService,
				portService: tt.fields.portService,
			}
			got, err := p.GetAll()
			if (err != nil) != tt.wantErr {
				t.Errorf("RoutingTableService.GetAll() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if !reflect.DeepEqual(got, tt.want) {
				t.Errorf("RoutingTableService.GetAll() = %v, want %v", got, tt.want)
			}
		})
	}
}
