package ports

import (
	"reflect"
	"testing"

	eventservice "code.fbi.h-da.de/danet/gosdn/controller/eventService"
	eventInterfaces "code.fbi.h-da.de/danet/gosdn/controller/interfaces/event"
	query "code.fbi.h-da.de/danet/gosdn/controller/store"
	"code.fbi.h-da.de/danet/gosdn/controller/topology/store"

	"github.com/google/uuid"
)

func getTestPort() Port {
	return Port{
		ID:   uuid.MustParse("44fb4aa4-c53c-4cf9-a081-5aabc61c7610"),
		Name: "Test-Port",
	}
}

func getEmptyPort() Port {
	return Port{
		ID:   uuid.Nil,
		Name: "",
	}
}

func getTestStoreWithPorts(t *testing.T, ports []Port) Store {
	store := store.NewGenericStore[Port]()

	for _, port := range ports {
		err := store.Add(port)
		if err != nil {
			t.Fatalf("failed to prepare test store while adding port: %v", err)
		}
	}

	return store
}

func TestNewPortService(t *testing.T) {
	type args struct {
		store        Store
		eventService eventInterfaces.Service
	}
	tests := []struct {
		name string
		args args
		want Service
	}{
		{
			name: "should create a new port service",
			args: args{
				store:        getTestStoreWithPorts(t, []Port{}),
				eventService: eventservice.NewMockEventService(),
			},
			want: NewPortService(getTestStoreWithPorts(t, []Port{}), eventservice.NewMockEventService()),
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			if got := NewPortService(tt.args.store, tt.args.eventService); !reflect.DeepEqual(got, tt.want) {
				t.Errorf("NewPortService() = %v, want %v", got, tt.want)
			}
		})
	}
}

func TestPortService_EnsureExists(t *testing.T) {
	type fields struct {
		store        Store
		eventService eventInterfaces.Service
	}
	type args struct {
		port Port
	}
	tests := []struct {
		name    string
		fields  fields
		args    args
		want    Port
		wantErr bool
	}{
		{
			name: "should not error if port with uuid is not in store",
			fields: fields{
				store:        store.NewGenericStore[Port](),
				eventService: eventservice.NewMockEventService(),
			},
			args: args{
				port: getTestPort(),
			},
			want:    getTestPort(),
			wantErr: false,
		},
		{
			name: "should return port that is in the store",
			fields: fields{
				store:        getTestStoreWithPorts(t, []Port{getTestPort()}),
				eventService: eventservice.NewMockEventService(),
			},
			args: args{
				port: getTestPort(),
			},
			want:    getTestPort(),
			wantErr: false,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			p := &PortService{
				store:        tt.fields.store,
				eventService: tt.fields.eventService,
			}
			got, err := p.EnsureExists(tt.args.port)
			if (err != nil) != tt.wantErr {
				t.Errorf("PortService.EnsureExists() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if !reflect.DeepEqual(got, tt.want) {
				t.Errorf("PortService.EnsureExists() = %v, want %v", got, tt.want)
			}
		})
	}
}

func TestPortService_Update(t *testing.T) {
	type fields struct {
		store        Store
		eventService eventInterfaces.Service
	}
	type args struct {
		port Port
	}
	tests := []struct {
		name    string
		fields  fields
		args    args
		want    Port
		wantErr bool
	}{
		{
			name: "should update an existing port",
			fields: fields{
				store:        getTestStoreWithPorts(t, []Port{getTestPort()}),
				eventService: eventservice.NewMockEventService(),
			},
			args: args{
				port: getTestPort(),
			},
			want:    getTestPort(),
			wantErr: false,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			p := &PortService{
				store:        tt.fields.store,
				eventService: tt.fields.eventService,
			}
			if err := p.Update(tt.args.port); (err != nil) != tt.wantErr {
				t.Errorf("PortService.Update() error = %v, wantErr %v", err, tt.wantErr)
			}

			updatedPort, err := p.Get(query.Query{ID: tt.args.port.ID})
			if err != nil {
				t.Errorf("PortService.Get() failed %v", err)
			}

			if !reflect.DeepEqual(updatedPort, tt.want) {
				t.Errorf("Got updated port = %v, want %v", updatedPort, tt.want)
			}
		})
	}
}

func TestPortService_Delete(t *testing.T) {
	type fields struct {
		store        Store
		eventService eventInterfaces.Service
	}
	type args struct {
		port Port
	}
	tests := []struct {
		name    string
		fields  fields
		args    args
		wantErr bool
	}{
		{
			name: "should delete an existing port",
			fields: fields{
				store:        getTestStoreWithPorts(t, []Port{getTestPort()}),
				eventService: eventservice.NewMockEventService(),
			},
			args: args{
				port: getTestPort(),
			},
			wantErr: false,
		},
		{
			name: "should fail if a port does not exists",
			fields: fields{
				store: store.NewGenericStore[Port](),
			},
			args: args{
				port: getTestPort(),
			},
			wantErr: true,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			p := &PortService{
				store:        tt.fields.store,
				eventService: tt.fields.eventService,
			}
			if err := p.Delete(tt.args.port); (err != nil) != tt.wantErr {
				t.Errorf("PortService.Delete() error = %v, wantErr %v", err, tt.wantErr)
			}
		})
	}
}

func TestPortService_Get(t *testing.T) {
	type fields struct {
		store        Store
		eventService eventInterfaces.Service
	}
	type args struct {
		query query.Query
	}
	tests := []struct {
		name    string
		fields  fields
		args    args
		want    Port
		wantErr bool
	}{
		{
			name: "should error if port with uuid is not in store",
			fields: fields{
				store: store.NewGenericStore[Port](),
			},
			args: args{
				query: query.Query{
					ID:   getTestPort().ID,
					Name: getTestPort().Name,
				},
			},
			want:    getEmptyPort(),
			wantErr: true,
		},
		{
			name: "should return port that is in the store",
			fields: fields{
				store:        getTestStoreWithPorts(t, []Port{getTestPort()}),
				eventService: eventservice.NewMockEventService(),
			},
			args: args{
				query: query.Query{
					ID:   getTestPort().ID,
					Name: getTestPort().Name,
				},
			},
			want:    getTestPort(),
			wantErr: false,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			p := &PortService{
				store:        tt.fields.store,
				eventService: tt.fields.eventService,
			}
			got, err := p.Get(tt.args.query)
			if (err != nil) != tt.wantErr {
				t.Errorf("PortService.Get() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if !reflect.DeepEqual(got, tt.want) {
				t.Errorf("PortService.Get() = %v, want %v", got, tt.want)
			}
		})
	}
}

func TestPortService_GetAll(t *testing.T) {
	type fields struct {
		store        Store
		eventService eventInterfaces.Service
	}
	tests := []struct {
		name    string
		fields  fields
		want    []Port
		wantErr bool
	}{
		{
			name: "should get all stored ports",
			fields: fields{
				store:        getTestStoreWithPorts(t, []Port{getTestPort()}),
				eventService: eventservice.NewMockEventService(),
			},
			want:    []Port{getTestPort()},
			wantErr: false,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			p := &PortService{
				store:        tt.fields.store,
				eventService: tt.fields.eventService,
			}
			got, err := p.GetAll()
			if (err != nil) != tt.wantErr {
				t.Errorf("PortService.GetAll() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if !reflect.DeepEqual(got, tt.want) {
				t.Errorf("PortService.GetAll() = %v, want %v", got, tt.want)
			}
		})
	}
}
