package nodes

import (
	"fmt"
	"time"

	"code.fbi.h-da.de/danet/gosdn/controller/customerrs"
	"code.fbi.h-da.de/danet/gosdn/controller/nucleus/database"
	query "code.fbi.h-da.de/danet/gosdn/controller/store"

	"github.com/google/uuid"
	"go.mongodb.org/mongo-driver/bson"
	"go.mongodb.org/mongo-driver/bson/primitive"
	"go.mongodb.org/mongo-driver/mongo"
	"go.mongodb.org/mongo-driver/mongo/options"
	"go.mongodb.org/mongo-driver/mongo/writeconcern"
)

// Store defines a NodeStore interface.
type Store interface {
	Add(Node) error
	Update(Node) error
	Delete(Node) error
	Get(query.Query) (Node, error)
	GetAll() ([]Node, error)
}

// DatabaseNodeStore is a database store for nodes.
type DatabaseNodeStore struct {
	storeName string
}

// NewDatabaseNodeStore returns a NodeStore.
func NewDatabaseNodeStore() Store {
	return &DatabaseNodeStore{
		storeName: fmt.Sprint("node-store.json"),
	}
}

// Get takes a nodes's UUID or name and returns the nodes.
func (s *DatabaseNodeStore) Get(query query.Query) (Node, error) {
	var loadedNode Node

	if query.ID.String() != "" {
		loadedNode, err := s.getByID(query.ID)
		if err != nil {
			return loadedNode, customerrs.CouldNotFindError{ID: query.ID, Name: query.Name}
		}

		return loadedNode, nil
	}

	loadedNode, err := s.getByName(query.Name)
	if err != nil {
		return loadedNode, customerrs.CouldNotFindError{ID: query.ID, Name: query.Name}
	}

	return loadedNode, nil
}

func (s *DatabaseNodeStore) getByID(idOfNode uuid.UUID) (loadedNode Node, err error) {
	client, ctx, cancel := database.GetMongoConnection()
	defer cancel()
	defer func() {
		if ferr := client.Disconnect(ctx); ferr != nil {
			fErrString := ferr.Error()
			err = fmt.Errorf("InternalError=%w DeferError=%+s", err, fErrString)
		}
	}()

	idAsByteArray, _ := idOfNode.MarshalBinary()

	db := client.Database(database.DatabaseName)
	collection := db.Collection(s.storeName)
	result := collection.FindOne(ctx, bson.D{primitive.E{Key: "_id", Value: idAsByteArray}})
	if result == nil {
		return loadedNode, customerrs.CouldNotFindError{ID: idOfNode}
	}

	err = result.Decode(&loadedNode)
	if err != nil {
		return loadedNode, customerrs.CouldNotMarshallError{Identifier: idOfNode, Type: loadedNode, Err: err}
	}

	return loadedNode, nil
}

func (s *DatabaseNodeStore) getByName(nameOfNode string) (loadedNode Node, err error) {
	client, ctx, cancel := database.GetMongoConnection()
	defer cancel()
	defer func() {
		if ferr := client.Disconnect(ctx); ferr != nil {
			fErrString := ferr.Error()
			err = fmt.Errorf("InternalError=%w DeferError=%+s", err, fErrString)
		}
	}()

	db := client.Database(database.DatabaseName)
	collection := db.Collection(s.storeName)
	result := collection.FindOne(ctx, bson.D{primitive.E{Key: "name", Value: nameOfNode}})
	if result == nil {
		return loadedNode, customerrs.CouldNotFindError{Name: nameOfNode}
	}

	err = result.Decode(&loadedNode)
	if err != nil {
		return loadedNode, customerrs.CouldNotMarshallError{Identifier: nameOfNode, Type: loadedNode, Err: err}
	}

	return loadedNode, nil
}

// GetAll returns all stored nodes.
func (s *DatabaseNodeStore) GetAll() (loadedNode []Node, err error) {
	client, ctx, cancel := database.GetMongoConnection()
	defer cancel()
	defer func() {
		if ferr := client.Disconnect(ctx); ferr != nil {
			fErrString := ferr.Error()
			err = fmt.Errorf("InternalError=%w DeferError=%+s", err, fErrString)
		}
	}()
	db := client.Database(database.DatabaseName)
	collection := db.Collection(s.storeName)

	cursor, err := collection.Find(ctx, bson.D{})
	if err != nil {
		return []Node{}, err
	}
	defer func() {
		if ferr := cursor.Close(ctx); ferr != nil {
			fErrString := ferr.Error()
			err = fmt.Errorf("InternalError=%w DeferError=%+s", err, fErrString)
		}
	}()

	err = cursor.All(ctx, &loadedNode)
	if err != nil {
		return loadedNode, customerrs.CouldNotMarshallError{Type: loadedNode, Err: err}
	}

	return loadedNode, nil
}

// Add adds a node to the node store.
func (s *DatabaseNodeStore) Add(node Node) (err error) {
	client, ctx, cancel := database.GetMongoConnection()
	defer cancel()
	defer func() {
		if ferr := client.Disconnect(ctx); ferr != nil {
			fErrString := ferr.Error()
			err = fmt.Errorf("InternalError=%w DeferError=%+s", err, fErrString)
		}
	}()

	node.Metadata.ResourceVersion = 0
	node.Metadata.CreatedAt = time.Now()
	node.Metadata.LastUpdated = time.Now()

	_, err = client.Database(database.DatabaseName).
		Collection(s.storeName).
		InsertOne(ctx, node)
	if err != nil {
		return customerrs.CouldNotCreateError{Identifier: node.ID, Type: node, Err: err}
	}

	return nil
}

// Update updates a existing node.
func (s *DatabaseNodeStore) Update(node Node) (err error) {
	var updatedLoadedNodes Node

	client, ctx, cancel := database.GetMongoConnection()
	defer cancel()
	defer func() {
		if ferr := client.Disconnect(ctx); ferr != nil {
			fErrString := ferr.Error()
			err = fmt.Errorf("InternalError=%w DeferError=%+s", err, fErrString)
		}
	}()

	// 1. Start Transaction
	wcMajority := writeconcern.New(writeconcern.WMajority())
	wcMajorityCollectionOpts := options.Collection().SetWriteConcern(wcMajority)
	nodeCollection := client.Database(database.DatabaseName).Collection(s.storeName, wcMajorityCollectionOpts)

	session, err := client.StartSession()
	if err != nil {
		return err
	}
	defer session.EndSession(ctx)

	// 2. Fetch exisiting Entity
	existingNode, err := s.getByID(node.ID)
	if err != nil {
		return err
	}

	// 3. Check if Entity.Metadata.ResourceVersion == UpdatedEntity.Metadata.ResourceVersion
	if node.Metadata.ResourceVersion != existingNode.Metadata.ResourceVersion {
		// 3.1.1 End transaction
		// 3.1.2 If no -> return error

		return fmt.Errorf(
			"resource version %d of provided node %s is older or newer than %d in the store",
			node.Metadata.ResourceVersion,
			node.ID.String(), existingNode.Metadata.ResourceVersion,
		)
	}

	// 3.2.1 If yes -> Update entity in callback
	callback := func(sessCtx mongo.SessionContext) (interface{}, error) {
		// Important: You must pass sessCtx as the Context parameter to the operations for them to be executed in the
		// transaction.
		update := bson.D{primitive.E{Key: "$set", Value: node}}

		upsert := false
		after := options.After
		opt := options.FindOneAndUpdateOptions{
			Upsert:         &upsert,
			ReturnDocument: &after,
		}

		err = nodeCollection.
			FindOneAndUpdate(
				ctx, bson.M{"_id": node.ID.String()}, update, &opt).
			Decode(&updatedLoadedNodes)
		if err != nil {
			return nil, customerrs.CouldNotUpdateError{Identifier: node.ID, Type: node, Err: err}
		}

		// 3.2.2 End transaction
		return "", nil
	}

	_, err = session.WithTransaction(ctx, callback)
	if err != nil {
		return err
	}

	return nil
}

// Delete deletes a node from the node store.
func (s *DatabaseNodeStore) Delete(node Node) (err error) {
	client, ctx, cancel := database.GetMongoConnection()
	defer cancel()
	defer func() {
		if ferr := client.Disconnect(ctx); ferr != nil {
			fErrString := ferr.Error()
			err = fmt.Errorf("InternalError=%w DeferError=%+s", err, fErrString)
		}
	}()

	db := client.Database(database.DatabaseName)
	collection := db.Collection(s.storeName)
	_, err = collection.DeleteOne(ctx, bson.D{primitive.E{Key: node.ID.String()}})
	if err != nil {
		return err
	}

	return nil
}
