package nodes

import (
	"reflect"
	"testing"

	"code.fbi.h-da.de/danet/gosdn/controller/conflict"
	eventservice "code.fbi.h-da.de/danet/gosdn/controller/eventService"
	"code.fbi.h-da.de/danet/gosdn/controller/interfaces/event"
	query "code.fbi.h-da.de/danet/gosdn/controller/store"
	"code.fbi.h-da.de/danet/gosdn/controller/topology/store"

	"github.com/google/uuid"
)

func getTestNode() Node {
	return Node{
		ID:   uuid.MustParse("44fb4aa4-c53c-4cf9-a081-5aabc61c7610"),
		Name: "Test-Node",
		Metadata: conflict.Metadata{
			ResourceVersion: 0,
		},
	}
}

func getEmptyNode() Node {
	return Node{
		ID:   uuid.Nil,
		Name: "",
	}
}

func getTestStoreWithNodes(t *testing.T, nodes []Node) Store {
	store := store.NewGenericStore[Node]()

	for _, node := range nodes {
		err := store.Add(node)
		if err != nil {
			t.Fatalf("failed to prepare test store while adding node: %v", err)
		}
	}

	return store
}

func TestNewNodeService(t *testing.T) {
	type args struct {
		store        Store
		eventService event.Service
	}
	tests := []struct {
		name string
		args args
		want Service
	}{
		{
			name: "should create a new node service",
			args: args{
				store:        getTestStoreWithNodes(t, []Node{}),
				eventService: eventservice.NewMockEventService(),
			},
			want: NewNodeService(getTestStoreWithNodes(t, []Node{}), eventservice.NewMockEventService()),
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			if got := NewNodeService(tt.args.store, tt.args.eventService); !reflect.DeepEqual(got, tt.want) {
				t.Errorf("NewNodeService() = %v, want %v", got, tt.want)
			}
		})
	}
}

func TestNodeService_EnsureExists(t *testing.T) {
	type fields struct {
		store        Store
		eventService event.Service
	}
	type args struct {
		node Node
	}
	tests := []struct {
		name    string
		fields  fields
		args    args
		want    Node
		wantErr bool
	}{
		{
			name: "should create if node with uuid is not in store",
			fields: fields{
				store:        store.NewGenericStore[Node](),
				eventService: eventservice.NewMockEventService(),
			},
			args: args{
				node: getTestNode(),
			},
			want:    getTestNode(),
			wantErr: false,
		},
		{
			name: "should return node that is in the store",
			fields: fields{
				store:        getTestStoreWithNodes(t, []Node{getTestNode()}),
				eventService: eventservice.NewMockEventService(),
			},
			args: args{
				node: getTestNode(),
			},
			want:    getTestNode(),
			wantErr: false,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			p := &NodeService{
				store:        tt.fields.store,
				eventService: tt.fields.eventService,
			}
			got, err := p.EnsureExists(tt.args.node)
			if (err != nil) != tt.wantErr {
				t.Errorf("NodeService.EnsureExists() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if !reflect.DeepEqual(got, tt.want) {
				t.Errorf("NodeService.EnsureExists() = %v, want %v", got, tt.want)
			}
		})
	}
}

func TestNodeService_Update(t *testing.T) {
	type fields struct {
		store        Store
		eventService event.Service
	}
	type args struct {
		node Node
	}
	tests := []struct {
		name    string
		fields  fields
		args    args
		want    Node
		wantErr bool
	}{
		{
			name: "should update an existing node",
			fields: fields{
				store:        getTestStoreWithNodes(t, []Node{getTestNode()}),
				eventService: eventservice.NewMockEventService(),
			},
			args: args{
				node: getTestNode(),
			},
			want:    getTestNode(),
			wantErr: false,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			p := &NodeService{
				store:        tt.fields.store,
				eventService: tt.fields.eventService,
			}
			if err := p.Update(tt.args.node); (err != nil) != tt.wantErr {
				t.Errorf("NodeService.Update() error = %v, wantErr %v", err, tt.wantErr)
			}

			updatedNode, err := p.Get(query.Query{ID: tt.args.node.ID, Name: tt.args.node.Name})
			if err != nil {
				t.Errorf("NodeService.Get() failed %v", err)
			}

			if !reflect.DeepEqual(updatedNode, tt.want) {
				t.Errorf("Got updated node = %v, want %v", updatedNode, tt.want)
			}
		})
	}
}

func TestNodeService_Delete(t *testing.T) {
	type fields struct {
		store        Store
		eventService event.Service
	}
	type args struct {
		node Node
	}
	tests := []struct {
		name    string
		fields  fields
		args    args
		wantErr bool
	}{
		{
			name: "should delete an existing node",
			fields: fields{
				store:        getTestStoreWithNodes(t, []Node{getTestNode()}),
				eventService: eventservice.NewMockEventService(),
			},
			args: args{
				node: getTestNode(),
			},
			wantErr: false,
		},
		{
			name: "should fail if a node does not exists",
			fields: fields{
				store:        store.NewGenericStore[Node](),
				eventService: eventservice.NewMockEventService(),
			},
			args: args{
				node: getTestNode(),
			},
			wantErr: true,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			p := &NodeService{
				store:        tt.fields.store,
				eventService: tt.fields.eventService,
			}
			if err := p.Delete(tt.args.node); (err != nil) != tt.wantErr {
				t.Errorf("NodeService.Delete() error = %v, wantErr %v", err, tt.wantErr)
			}
		})
	}
}

func TestNodeService_Get(t *testing.T) {
	type fields struct {
		store        Store
		eventService event.Service
	}
	type args struct {
		query query.Query
	}
	tests := []struct {
		name    string
		fields  fields
		args    args
		want    Node
		wantErr bool
	}{
		{
			name: "should error if node with uuid is not in store",
			fields: fields{
				store:        store.NewGenericStore[Node](),
				eventService: eventservice.NewMockEventService(),
			},
			args: args{
				query: query.Query{
					ID:   getTestNode().ID,
					Name: getTestNode().Name,
				},
			},
			want:    getEmptyNode(),
			wantErr: true,
		},
		{
			name: "should return node that is in the store",
			fields: fields{
				store:        getTestStoreWithNodes(t, []Node{getTestNode()}),
				eventService: eventservice.NewMockEventService(),
			},
			args: args{
				query: query.Query{
					ID:   getTestNode().ID,
					Name: getTestNode().Name,
				},
			},
			want:    getTestNode(),
			wantErr: false,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			p := &NodeService{
				store:        tt.fields.store,
				eventService: tt.fields.eventService,
			}
			got, err := p.Get(tt.args.query)
			if (err != nil) != tt.wantErr {
				t.Errorf("NodeService.Get() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if !reflect.DeepEqual(got, tt.want) {
				t.Errorf("NodeService.Get() = %v, want %v", got, tt.want)
			}
		})
	}
}

func TestNodeService_GetAll(t *testing.T) {
	type fields struct {
		store        Store
		eventService event.Service
	}
	tests := []struct {
		name    string
		fields  fields
		want    []Node
		wantErr bool
	}{
		{
			name: "should get all stored nodes",
			fields: fields{
				store:        getTestStoreWithNodes(t, []Node{getTestNode()}),
				eventService: eventservice.NewMockEventService(),
			},
			want:    []Node{getTestNode()},
			wantErr: false,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			p := &NodeService{
				store:        tt.fields.store,
				eventService: tt.fields.eventService,
			}
			got, err := p.GetAll()
			if (err != nil) != tt.wantErr {
				t.Errorf("NodeService.GetAll() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if !reflect.DeepEqual(got, tt.want) {
				t.Errorf("NodeService.GetAll() = %v, want %v", got, tt.want)
			}
		})
	}
}
