package rbac

import (
	"reflect"
	"testing"

	"code.fbi.h-da.de/danet/gosdn/controller/conflict"
	"code.fbi.h-da.de/danet/gosdn/controller/interfaces/rbac"
	"code.fbi.h-da.de/danet/gosdn/controller/store"
	"github.com/google/uuid"
)

func TestFileSystemUserStore_Add(t *testing.T) {
	defer ensureStoreFileForTestsIsRemoved(store.UserFilename)

	type args struct {
		UserToAdd rbac.User
	}
	var idtest uuid.UUID
	var role map[string]string
	testingUser := NewUser(idtest, "testUser", role, "xyz", "svsvsfbdwbwbev", "svswvasfbw", conflict.Metadata{})
	tests := []struct {
		name    string
		args    args
		wantErr bool
	}{
		{
			"testUser",
			args{testingUser},
			false,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			s := NewUserStore()

			if err := s.Add(tt.args.UserToAdd); (err != nil) != tt.wantErr {
				t.Errorf("FileSystemUserStore.Add() error = %v, wantErr %v", err, tt.wantErr)
			}
		})
	}
}

func TestFileSystemUserStore_Delete(t *testing.T) {
	defer ensureStoreFileForTestsIsRemoved(store.UserFilename)

	type args struct {
		UserToDelete rbac.User
	}
	var idtest uuid.UUID
	var role map[string]string
	testingUser := NewUser(idtest, "", role, "xyz", "svsvsfbdwbwbev", "svswvasfbw", conflict.Metadata{})
	tests := []struct {
		name    string
		args    args
		wantErr bool
	}{
		{
			"testUser",
			args{testingUser},
			false,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			s := NewUserStore()
			if err := s.Add(testingUser); err != nil {
				t.Error(err)
			}
			if err := s.Delete(tt.args.UserToDelete); (err != nil) != tt.wantErr {
				t.Errorf("FileSystemUserStore.Delete() error = %v, wantErr %v", err, tt.wantErr)
			}
		})
	}
}

func TestFileSystemUserStore_Get(t *testing.T) {
	defer ensureStoreFileForTestsIsRemoved(store.UserFilename)

	type args struct {
		query store.Query
	}
	var idtest uuid.UUID
	var role map[string]string
	testingUser := NewUser(idtest, "", role, "xyz", "svsvsfbdwbwbev", "svswvasfbw", conflict.Metadata{})
	tests := []struct {
		name    string
		args    args
		want    rbac.LoadedUser
		wantErr bool
	}{
		{
			"testUser",
			args{
				store.Query{ID: idtest, Name: "test"},
			},
			rbac.LoadedUser{ID: idtest.String(), UserName: "", Roles: role, Password: "xyz", Token: "svsvsfbdwbwbev", Salt: "svswvasfbw"},
			false,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			s := NewUserStore()
			if err := s.Add(testingUser); err != nil {
				t.Error(err)
			}
			got, err := s.Get(tt.args.query)
			if (err != nil) != tt.wantErr {
				t.Errorf("FileSystemUserStore.Get() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if !reflect.DeepEqual(got, tt.want) {
				t.Errorf("FileSystemUserStore.Get() = %v, want %v", got, tt.want)
			}
		})
	}
}

func TestFileSystemUserStore_GetAll(t *testing.T) {
	defer ensureStoreFileForTestsIsRemoved(store.UserFilename)

	var idtest uuid.UUID
	var role map[string]string
	testingUser1 := NewUser(idtest, "", role, "xyz", "svsvsfbdwbwbevasf", "svswvasfbwasv", conflict.Metadata{})
	testingUser2 := NewUser(idtest, "", role, "abc", "svsvsfbdwbwbevsav", "svswvasfbwadf", conflict.Metadata{})
	testingUser3 := NewUser(idtest, "", role, "lmn", "svsvsfbdwbwbevscv", "svswvasfbwasd", conflict.Metadata{})
	tests := []struct {
		name    string
		want    []rbac.LoadedUser
		wantErr bool
	}{
		{
			"testUser",
			[]rbac.LoadedUser{{ID: idtest.String(), UserName: "", Roles: role, Password: "xyz", Token: "svsvsfbdwbwbevasf", Salt: "svswvasfbwasv"}, {ID: idtest.String(), UserName: "", Roles: role, Password: "abc", Token: "svsvsfbdwbwbevsav", Salt: "svswvasfbwadf"}, {ID: idtest.String(), UserName: "", Roles: role, Password: "lmn", Token: "svsvsfbdwbwbevscv", Salt: "svswvasfbwasd"}},
			false,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			var errs []error
			s := NewUserStore()
			if err := s.Add(testingUser1); err != nil {
				errs = append(errs, err)
			}
			if err := s.Add(testingUser2); err != nil {
				errs = append(errs, err)
			}
			if err := s.Add(testingUser3); err != nil {
				errs = append(errs, err)
			}

			if len(errs) > 0 {
				t.Error(errs)
			}

			got, err := s.GetAll()
			if (err != nil) != tt.wantErr {
				t.Errorf("FileSystemUserStore.GetAll() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if !reflect.DeepEqual(got, tt.want) {
				t.Errorf("FileSystemUserStore.GetAll() = %v, want %v", got, tt.want)
			}
		})
	}
}

func TestFileSystemUserStore_Update(t *testing.T) {
	defer ensureStoreFileForTestsIsRemoved(store.UserFilename)

	type args struct {
		userToUpdate rbac.User
	}
	var idtest uuid.UUID
	var role map[string]string
	testingUser := NewUser(idtest, "", role, "xyz", "svsvsfbdwbwbev", "svswvasfbw", conflict.Metadata{})
	tests := []struct {
		name    string
		args    args
		wantErr bool
	}{
		{
			"testUser",
			args{
				testingUser,
			},
			false,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			s := NewUserStore()
			if err := s.Add(testingUser); err != nil {
				t.Error(err)
			}
			if err := s.Update(tt.args.userToUpdate); (err != nil) != tt.wantErr {
				t.Errorf("FileSystemUserStore.Update() error = %v, wantErr %v", err, tt.wantErr)
			}
		})
	}
}
