package rbac

import (
	"encoding/json"

	"code.fbi.h-da.de/danet/gosdn/controller/customerrs"
	"code.fbi.h-da.de/danet/gosdn/controller/interfaces/rbac"
	"code.fbi.h-da.de/danet/gosdn/controller/store"
)

// MemoryUserStore provides a in-memory implementation for users.
type MemoryUserStore struct {
	Store           map[string]rbac.LoadedUser
	nameLookupTable map[string]string
}

// NewMemoryUserStore returns a specific in-memory store for users.
func NewMemoryUserStore() rbac.UserStore {
	return &MemoryUserStore{
		Store:           make(map[string]rbac.LoadedUser),
		nameLookupTable: make(map[string]string),
	}
}

// Add adds a item to the store.
func (s *MemoryUserStore) Add(item rbac.User) error {
	var user rbac.LoadedUser

	b, err := json.Marshal(item)
	if err != nil {
		return err
	}
	err = json.Unmarshal(b, &user)
	if err != nil {
		return err
	}

	_, ok := s.Store[user.ID]
	if ok {
		return nil
	}

	s.Store[user.ID] = user
	s.nameLookupTable[item.Name()] = user.ID

	return nil
}

// Delete deletes a user from the user store.
func (s *MemoryUserStore) Delete(item rbac.User) error {
	delete(s.Store, item.ID().String())
	return nil
}

// Update updates an existing user.
func (s *MemoryUserStore) Update(item rbac.User) error {
	_, ok := s.Store[item.ID().String()]
	if !ok {
		return customerrs.CouldNotFindError{ID: item.ID(), Name: item.Name()}
	}

	var user rbac.LoadedUser

	b, err := json.Marshal(item)
	if err != nil {
		return err
	}
	err = json.Unmarshal(b, &user)
	if err != nil {
		return err
	}

	s.Store[item.ID().String()] = user
	s.nameLookupTable[item.Name()] = item.ID().String()

	return nil
}

// Get takes a user's UUID or name and returns the user.
func (s *MemoryUserStore) Get(query store.Query) (rbac.LoadedUser, error) {
	// First search for direct hit on UUID.
	item, ok := s.Store[query.ID.String()]
	if !ok {
		// Second search for name
		id, ok := s.nameLookupTable[query.Name]
		if !ok {
			return item, customerrs.CouldNotFindError{ID: query.ID, Name: query.Name}
		}

		item, ok := s.Store[id]
		if !ok {
			return item, customerrs.CouldNotFindError{ID: query.ID, Name: query.Name}
		}

		return item, nil
	}

	return item, nil
}

// GetAll returns all stored users.
func (s *MemoryUserStore) GetAll() ([]rbac.LoadedUser, error) {
	var allItems []rbac.LoadedUser

	for _, item := range s.Store {
		allItems = append(allItems, item)
	}

	return allItems, nil
}
