package rbac

import (
	"encoding/json"

	"code.fbi.h-da.de/danet/gosdn/controller/customerrs"
	"code.fbi.h-da.de/danet/gosdn/controller/interfaces/rbac"
	"code.fbi.h-da.de/danet/gosdn/controller/store"
)

// MemoryRoleStore provides a in-memory implementation for roles.
type MemoryRoleStore struct {
	Store           map[string]rbac.LoadedRole
	nameLookupTable map[string]string
}

// NewMemoryRoleStore returns a specific in-memory store for roles.
func NewMemoryRoleStore() rbac.RoleStore {
	return &MemoryRoleStore{
		Store:           make(map[string]rbac.LoadedRole),
		nameLookupTable: make(map[string]string),
	}
}

// Add adds a item to the store.
func (s *MemoryRoleStore) Add(item rbac.Role) error {
	var role rbac.LoadedRole

	b, err := json.Marshal(item)
	if err != nil {
		return err
	}
	err = json.Unmarshal(b, &role)
	if err != nil {
		return err
	}

	_, ok := s.Store[role.ID]
	if ok {
		return nil
	}

	s.Store[role.ID] = role
	s.nameLookupTable[item.Name()] = role.ID

	return nil
}

// Delete deletes a role from the role store.
func (s *MemoryRoleStore) Delete(item rbac.Role) error {
	delete(s.Store, item.ID().String())
	return nil
}

// Update updates an existing role.
func (s *MemoryRoleStore) Update(item rbac.Role) error {
	_, ok := s.Store[item.ID().String()]
	if !ok {
		return customerrs.CouldNotFindError{ID: item.ID(), Name: item.Name()}
	}

	var role rbac.LoadedRole

	b, err := json.Marshal(item)
	if err != nil {
		return err
	}
	err = json.Unmarshal(b, &role)
	if err != nil {
		return err
	}

	s.Store[item.ID().String()] = role
	s.nameLookupTable[item.Name()] = item.ID().String()

	return nil
}

// Get takes a role's UUID or name and returns the role.
func (s *MemoryRoleStore) Get(query store.Query) (rbac.LoadedRole, error) {
	// First search for direct hit on UUID.
	item, ok := s.Store[query.ID.String()]
	if !ok {
		// Second search for name
		id, ok := s.nameLookupTable[query.Name]
		if !ok {
			return item, customerrs.CouldNotFindError{ID: query.ID, Name: query.Name}
		}

		item, ok := s.Store[id]
		if !ok {
			return item, customerrs.CouldNotFindError{ID: query.ID, Name: query.Name}
		}

		return item, nil
	}

	return item, nil
}

// GetAll returns all stored roles.
func (s *MemoryRoleStore) GetAll() ([]rbac.LoadedRole, error) {
	var allItems []rbac.LoadedRole

	for _, item := range s.Store {
		allItems = append(allItems, item)
	}

	return allItems, nil
}
