package rbac

import (
	"fmt"

	"code.fbi.h-da.de/danet/gosdn/controller/customerrs"
	"code.fbi.h-da.de/danet/gosdn/controller/interfaces/rbac"
	"code.fbi.h-da.de/danet/gosdn/controller/nucleus/database"
	"code.fbi.h-da.de/danet/gosdn/controller/store"
	"github.com/google/uuid"
	log "github.com/sirupsen/logrus"
	"go.mongodb.org/mongo-driver/bson"
	"go.mongodb.org/mongo-driver/bson/primitive"
	"go.mongodb.org/mongo-driver/mongo"
	"go.mongodb.org/mongo-driver/mongo/options"
	"go.mongodb.org/mongo-driver/mongo/writeconcern"
)

// DatabaseUserStore is used to store users in database.
type DatabaseUserStore struct {
	userStoreName string
}

// Add adds an User.
func (s *DatabaseUserStore) Add(userToAdd rbac.User) (err error) {
	client, ctx, cancel := database.GetMongoConnection()
	defer cancel()
	defer func() {
		if ferr := client.Disconnect(ctx); ferr != nil {
			fErrString := ferr.Error()
			err = fmt.Errorf("InternalError=%w DeferError=%+s", err, fErrString)
		}
	}()

	_, err = client.Database(database.DatabaseName).
		Collection(s.userStoreName).
		InsertOne(ctx, userToAdd)
	if err != nil {
		if mongo.IsDuplicateKeyError(err) {
			return nil
		}

		return customerrs.CouldNotCreateError{Identifier: userToAdd.ID(), Type: userToAdd, Err: err}
	}

	return nil
}

// Delete deletes an User.
func (s *DatabaseUserStore) Delete(userToDelete rbac.User) (err error) {
	client, ctx, cancel := database.GetMongoConnection()
	defer cancel()
	defer func() {
		if ferr := client.Disconnect(ctx); ferr != nil {
			fErrString := ferr.Error()
			err = fmt.Errorf("InternalError=%w DeferError=%+s", err, fErrString)
		}
	}()

	_, err = client.Database(database.DatabaseName).
		Collection(s.userStoreName).
		DeleteOne(ctx, bson.D{primitive.E{Key: "_id", Value: userToDelete.ID().String()}})
	if err != nil {
		return customerrs.CouldNotDeleteError{Identifier: userToDelete.ID(), Type: userToDelete, Err: err}
	}

	return nil
}

// Get takes a User's UUID or name and returns the User. If the requested
// User does not exist an error is returned.
func (s *DatabaseUserStore) Get(query store.Query) (rbac.LoadedUser, error) {
	var loadedUser rbac.LoadedUser

	if query.ID != uuid.Nil {
		loadedUser, err := s.getByID(query.ID)
		if err != nil {
			return loadedUser, err
		}

		return loadedUser, nil
	}

	loadedUser, err := s.getByName(query.Name)
	if err != nil {
		return loadedUser, err
	}

	return loadedUser, nil
}

func (s *DatabaseUserStore) getByID(idOfUser uuid.UUID) (loadedUser rbac.LoadedUser, err error) {
	client, ctx, cancel := database.GetMongoConnection()
	defer cancel()
	defer func() {
		if ferr := client.Disconnect(ctx); ferr != nil {
			fErrString := ferr.Error()
			err = fmt.Errorf("InternalError=%w DeferError=%+s", err, fErrString)
		}
	}()

	db := client.Database(database.DatabaseName)
	collection := db.Collection(s.userStoreName)
	result := collection.FindOne(ctx, bson.D{primitive.E{Key: "_id", Value: idOfUser.String()}})
	if result == nil {
		return loadedUser, customerrs.CouldNotFindError{ID: idOfUser}
	}

	err = result.Decode(&loadedUser)
	if err != nil {
		log.Printf("Failed marshalling %v", err)
		return loadedUser, customerrs.CouldNotMarshallError{Identifier: idOfUser, Type: loadedUser, Err: err}
	}

	return loadedUser, nil
}

func (s *DatabaseUserStore) getByName(nameOfUser string) (loadedUser rbac.LoadedUser, err error) {
	client, ctx, cancel := database.GetMongoConnection()
	defer cancel()
	defer func() {
		if ferr := client.Disconnect(ctx); ferr != nil {
			fErrString := ferr.Error()
			err = fmt.Errorf("InternalError=%w DeferError=%+s", err, fErrString)
		}
	}()

	db := client.Database(database.DatabaseName)
	collection := db.Collection(s.userStoreName)
	result := collection.FindOne(ctx, bson.D{primitive.E{Key: "username", Value: nameOfUser}})
	if result == nil {
		return loadedUser, customerrs.CouldNotFindError{Name: nameOfUser}
	}

	err = result.Decode(&loadedUser)
	if err != nil {
		log.Printf("Failed marshalling %v", err)
		return loadedUser, customerrs.CouldNotMarshallError{Identifier: nameOfUser, Type: loadedUser, Err: err}
	}

	return loadedUser, nil
}

// GetAll returns all Users.
func (s *DatabaseUserStore) GetAll() (loadedUsers []rbac.LoadedUser, err error) {
	client, ctx, cancel := database.GetMongoConnection()
	defer cancel()
	defer func() {
		if ferr := client.Disconnect(ctx); ferr != nil {
			fErrString := ferr.Error()
			err = fmt.Errorf("InternalError=%w DeferError=%+s", err, fErrString)
		}
	}()

	db := client.Database(database.DatabaseName)
	collection := db.Collection(s.userStoreName)

	cursor, err := collection.Find(ctx, bson.D{})
	if err != nil {
		return nil, err
	}
	defer func() {
		if ferr := cursor.Close(ctx); ferr != nil {
			fErrString := ferr.Error()
			err = fmt.Errorf("InternalError=%w DeferError=%+s", err, fErrString)
		}
	}()

	err = cursor.All(ctx, &loadedUsers)
	if err != nil {
		log.Printf("Failed marshalling %v", err)

		return nil, customerrs.CouldNotMarshallError{Type: loadedUsers, Err: err}
	}
	return loadedUsers, nil
}

// Update updates the User.
func (s *DatabaseUserStore) Update(userToUpdate rbac.User) (err error) {
	var updatedLoadedUser rbac.LoadedUser

	client, ctx, cancel := database.GetMongoConnection()
	defer cancel()
	defer func() {
		if ferr := client.Disconnect(ctx); ferr != nil {
			fErrString := ferr.Error()
			err = fmt.Errorf("InternalError=%w DeferError=%+s", err, fErrString)
		}
	}()

	// 1. Start Transaction
	wcMajority := writeconcern.New(writeconcern.WMajority())
	wcMajorityCollectionOpts := options.Collection().SetWriteConcern(wcMajority)
	userCollection := client.Database(database.DatabaseName).Collection(s.userStoreName, wcMajorityCollectionOpts)

	session, err := client.StartSession()
	if err != nil {
		return err
	}
	defer session.EndSession(ctx)

	// 2. Fetch exisiting Entity
	existingUser, err := s.getByID(userToUpdate.ID())
	if err != nil {
		return err
	}

	// 3. Check if Entity.Metadata.ResourceVersion == UpdatedEntity.Metadata.ResourceVersion
	if userToUpdate.GetMetadata().ResourceVersion != existingUser.Metadata.ResourceVersion {
		// 3.1.1 End transaction
		// 3.1.2 If no -> return error

		return fmt.Errorf(
			"resource version %d of provided user %s is older or newer than %d in the store",
			userToUpdate.GetMetadata().ResourceVersion,
			userToUpdate.ID().String(), existingUser.Metadata.ResourceVersion,
		)
	}

	// 3.2.1 If yes -> Update entity in callback
	callback := func(sessCtx mongo.SessionContext) (interface{}, error) {
		// Important: You must pass sessCtx as the Context parameter to the operations for them to be executed in the
		// transaction.

		u, _ := userToUpdate.(*User)
		u.Metadata.ResourceVersion = u.Metadata.ResourceVersion + 1

		update := bson.D{primitive.E{Key: "$set", Value: u}}

		upsert := false
		after := options.After
		opt := options.FindOneAndUpdateOptions{
			Upsert:         &upsert,
			ReturnDocument: &after,
		}

		err = userCollection.
			FindOneAndUpdate(
				ctx, bson.M{"_id": userToUpdate.ID().String()}, update, &opt).
			Decode(&updatedLoadedUser)
		if err != nil {
			log.Printf("Could not update User: %v", err)

			return nil, customerrs.CouldNotUpdateError{Identifier: userToUpdate.ID(), Type: userToUpdate, Err: err}
		}

		// 3.2.2 End transaction
		return "", nil
	}

	_, err = session.WithTransaction(ctx, callback)
	if err != nil {
		return err
	}

	return nil
}
