package rbac

import (
	"fmt"

	"code.fbi.h-da.de/danet/gosdn/controller/customerrs"
	"code.fbi.h-da.de/danet/gosdn/controller/interfaces/rbac"
	"code.fbi.h-da.de/danet/gosdn/controller/nucleus/database"
	"code.fbi.h-da.de/danet/gosdn/controller/store"
	"github.com/google/uuid"
	log "github.com/sirupsen/logrus"
	"go.mongodb.org/mongo-driver/bson"
	"go.mongodb.org/mongo-driver/bson/primitive"
	"go.mongodb.org/mongo-driver/mongo"
	"go.mongodb.org/mongo-driver/mongo/options"
)

// DatabaseRoleStore is used to store roles in database.
type DatabaseRoleStore struct {
	roleStoreName string
}

// Add adds a Role.
func (s *DatabaseRoleStore) Add(roleToAdd rbac.Role) (err error) {
	client, ctx, cancel := database.GetMongoConnection()
	defer cancel()
	defer func() {
		if ferr := client.Disconnect(ctx); ferr != nil {
			fErrString := ferr.Error()
			err = fmt.Errorf("InternalError=%w DeferError=%+s", err, fErrString)
		}
	}()

	_, err = client.Database(database.DatabaseName).
		Collection(s.roleStoreName).
		InsertOne(ctx, roleToAdd)
	if err != nil {
		if mongo.IsDuplicateKeyError(err) {
			return nil
		}

		return customerrs.CouldNotCreateError{Identifier: roleToAdd.ID(), Type: roleToAdd, Err: err}
	}

	return nil
}

// Delete deletes a Role.
func (s *DatabaseRoleStore) Delete(roleToDelete rbac.Role) (err error) {
	client, ctx, cancel := database.GetMongoConnection()
	defer cancel()
	defer func() {
		if ferr := client.Disconnect(ctx); ferr != nil {
			fErrString := ferr.Error()
			err = fmt.Errorf("InternalError=%w DeferError=%+s", err, fErrString)
		}
	}()

	_, err = client.Database(database.DatabaseName).
		Collection(s.roleStoreName).
		DeleteOne(ctx, bson.D{primitive.E{Key: "_id", Value: roleToDelete.ID().String()}})
	if err != nil {
		return customerrs.CouldNotDeleteError{Identifier: roleToDelete.ID(), Type: roleToDelete, Err: err}
	}

	return nil
}

// Get takes a Roles's UUID or name and returns the Role. If the requested
// Role does not exist an error is returned.
func (s *DatabaseRoleStore) Get(query store.Query) (rbac.LoadedRole, error) {
	var loadedRole rbac.LoadedRole

	if query.ID != uuid.Nil {
		loadedRole, err := s.getByID(query.ID)
		if err != nil {
			return loadedRole, err
		}

		return loadedRole, nil
	}

	loadedRole, err := s.getByName(query.Name)
	if err != nil {
		return loadedRole, err
	}

	return loadedRole, nil
}

func (s *DatabaseRoleStore) getByID(idOfRole uuid.UUID) (loadedRole rbac.LoadedRole, err error) {
	client, ctx, cancel := database.GetMongoConnection()
	defer cancel()
	defer func() {
		if ferr := client.Disconnect(ctx); ferr != nil {
			fErrString := ferr.Error()
			err = fmt.Errorf("InternalError=%w DeferError=%+s", err, fErrString)
		}
	}()

	db := client.Database(database.DatabaseName)
	collection := db.Collection(s.roleStoreName)
	result := collection.FindOne(ctx, bson.D{primitive.E{Key: "_id", Value: idOfRole.String()}})
	if result == nil {
		return loadedRole, customerrs.CouldNotFindError{ID: idOfRole}
	}

	err = result.Decode(&loadedRole)
	if err != nil {
		log.Printf("Failed marshalling %v", err)
		return loadedRole, customerrs.CouldNotMarshallError{Identifier: idOfRole, Type: loadedRole, Err: err}
	}

	return loadedRole, nil
}

func (s *DatabaseRoleStore) getByName(nameOfRole string) (loadedRole rbac.LoadedRole, err error) {
	client, ctx, cancel := database.GetMongoConnection()
	defer cancel()
	defer func() {
		if ferr := client.Disconnect(ctx); ferr != nil {
			fErrString := ferr.Error()
			err = fmt.Errorf("InternalError=%w DeferError=%+s", err, fErrString)
		}
	}()

	db := client.Database(database.DatabaseName)
	collection := db.Collection(s.roleStoreName)
	result := collection.FindOne(ctx, bson.D{primitive.E{Key: "rolename", Value: nameOfRole}})
	if result == nil {
		return loadedRole, customerrs.CouldNotFindError{Name: nameOfRole}
	}

	err = result.Decode(&loadedRole)
	if err != nil {
		log.Printf("Failed marshalling %v", err)
		return loadedRole, customerrs.CouldNotMarshallError{Identifier: nameOfRole, Type: loadedRole, Err: err}
	}

	return loadedRole, nil
}

// GetAll returns all Roles.
func (s *DatabaseRoleStore) GetAll() (loadedRoles []rbac.LoadedRole, err error) {
	client, ctx, cancel := database.GetMongoConnection()
	defer cancel()
	defer func() {
		if ferr := client.Disconnect(ctx); ferr != nil {
			fErrString := ferr.Error()
			err = fmt.Errorf("InternalError=%w DeferError=%+s", err, fErrString)
		}
	}()

	db := client.Database(database.DatabaseName)
	collection := db.Collection(s.roleStoreName)

	cursor, err := collection.Find(ctx, bson.D{})
	if err != nil {
		return nil, err
	}
	defer func() {
		if ferr := cursor.Close(ctx); ferr != nil {
			fErrString := ferr.Error()
			err = fmt.Errorf("InternalError=%w DeferError=%+s", err, fErrString)
		}
	}()

	err = cursor.All(ctx, &loadedRoles)
	if err != nil {
		log.Printf("Failed marshalling %v", err)

		return nil, customerrs.CouldNotMarshallError{Type: loadedRoles, Err: err}
	}
	return loadedRoles, nil
}

// Update updates the role.
func (s *DatabaseRoleStore) Update(roleToUpdate rbac.Role) (err error) {
	var updatedLoadedRole rbac.LoadedRole

	client, ctx, cancel := database.GetMongoConnection()
	defer cancel()
	defer func() {
		if ferr := client.Disconnect(ctx); ferr != nil {
			fErrString := ferr.Error()
			err = fmt.Errorf("InternalError=%w DeferError=%+s", err, fErrString)
		}
	}()
	update := bson.D{primitive.E{Key: "$set", Value: roleToUpdate}}

	upsert := false
	after := options.After
	opt := options.FindOneAndUpdateOptions{
		Upsert:         &upsert,
		ReturnDocument: &after,
	}

	err = client.Database(database.DatabaseName).
		Collection(s.roleStoreName).
		FindOneAndUpdate(
			ctx, bson.M{"_id": roleToUpdate.ID().String()}, update, &opt).
		Decode(&updatedLoadedRole)
	if err != nil {
		log.Printf("Could not update Role: %v", err)

		return customerrs.CouldNotUpdateError{Identifier: roleToUpdate.ID(), Type: roleToUpdate, Err: err}
	}

	return nil
}
