package nucleus

import (
	"testing"

	tpb "code.fbi.h-da.de/danet/gosdn/api/go/gosdn/transport"
	"code.fbi.h-da.de/danet/gosdn/controller/interfaces/plugin"
	gpb "github.com/openconfig/gnmi/proto/gnmi"
)

// TestNewTransport is for input validation only. Functional tests
// are conducted at the transport implementation constructors.
func TestNewTransport(t *testing.T) {
	type args struct {
		opts   *tpb.TransportOption
		plugin plugin.Plugin
	}
	tests := []struct {
		name    string
		args    args
		wantErr bool
	}{
		{
			name: "default",
			args: args{
				opts: &tpb.TransportOption{
					Address:  "test",
					Username: "test",
					Password: "test",
					Tls:      false,
					TransportOption: &tpb.TransportOption_GnmiTransportOption{
						GnmiTransportOption: &tpb.GnmiTransportOption{
							Encoding: gpb.Encoding_PROTO,
						},
					},
				},
				plugin: mockPlugin(t),
			},
			wantErr: false,
		},
		{
			name: "no opt",
			args: args{
				opts:   nil,
				plugin: mockPlugin(t),
			},
			wantErr: true,
		},
		{
			name: "no sbi",
			args: args{
				opts: &tpb.TransportOption{
					Address:  "test",
					Username: "test",
					Password: "test",
					Tls:      false,
					TransportOption: &tpb.TransportOption_GnmiTransportOption{
						GnmiTransportOption: &tpb.GnmiTransportOption{
							Encoding: gpb.Encoding_PROTO,
						},
					},
				},
				plugin: nil,
			},
			wantErr: true,
		},
		{
			name: "no implementation options",
			args: args{
				opts: &tpb.TransportOption{
					Address:  "test",
					Username: "test",
					Password: "test",
					Tls:      false,
				},
				plugin: mockPlugin(t),
			},
			wantErr: true,
		},
		{
			name: "no inner implementation options",
			args: args{
				opts: &tpb.TransportOption{
					Address:         "test",
					Username:        "test",
					Password:        "test",
					Tls:             false,
					TransportOption: &tpb.TransportOption_GnmiTransportOption{},
				},
				plugin: mockPlugin(t),
			},
			wantErr: true,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			_, err := NewTransport(tt.args.opts, tt.args.plugin)
			if (err != nil) != tt.wantErr {
				t.Errorf("NewTransport() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
		})
	}
}
