package nucleus

import (
	"encoding/json"

	"code.fbi.h-da.de/danet/gosdn/controller/customerrs"
	"code.fbi.h-da.de/danet/gosdn/controller/interfaces/plugin"
	"code.fbi.h-da.de/danet/gosdn/controller/store"
)

// MemoryPluginStore provides a in-memory implementation for plugins.
type MemoryPluginStore struct {
	Store           map[string]plugin.LoadedPlugin
	nameLookupTable map[string]string
}

// NewMemoryPluginStore returns a specific in-memory store for plugins.
func NewMemoryPluginStore() plugin.Store {
	return &MemoryPluginStore{
		Store:           make(map[string]plugin.LoadedPlugin),
		nameLookupTable: make(map[string]string),
	}
}

// Add adds a item to the store.
func (t *MemoryPluginStore) Add(item plugin.Plugin) error {
	loadedPlugin, err := store.TransformObjectToLoadedObject[plugin.Plugin, plugin.LoadedPlugin](item)
	if err != nil {
		return err
	}

	_, ok := t.Store[loadedPlugin.ID]
	if ok {
		return nil
	}

	t.Store[loadedPlugin.ID] = loadedPlugin

	return nil
}

// Update updates a existing plugin.
func (t *MemoryPluginStore) Update(item plugin.Plugin) error {
	_, ok := t.Store[item.ID().String()]
	if ok {
		return nil
	}

	var plugin plugin.LoadedPlugin

	b, err := json.Marshal(item)
	if err != nil {
		return err
	}
	err = json.Unmarshal(b, &plugin)
	if err != nil {
		return err
	}

	t.Store[item.ID().String()] = plugin
	//TODO: add name to plugin
	//t.nameLookupTable[item.Name()] = item.ID().String()

	return nil
}

// Delete deletes a plugin from the store.
func (t *MemoryPluginStore) Delete(item plugin.Plugin) error {
	delete(t.Store, item.ID().String())

	return nil
}

// Get takes a plugins's UUID or name and returns the plugin.
func (t *MemoryPluginStore) Get(query store.Query) (plugin.LoadedPlugin, error) {
	// First search for direct hit on UUID.
	item, ok := t.Store[query.ID.String()]
	if !ok {
		// Second search for name
		id, ok := t.nameLookupTable[query.Name]
		if !ok {
			return item, customerrs.CouldNotFindError{ID: query.ID, Name: query.Name}
		}

		item, ok := t.Store[id]
		if !ok {
			return item, customerrs.CouldNotFindError{ID: query.ID, Name: query.Name}
		}

		return item, nil
	}

	return item, nil
}

// GetAll returns all stored plugins.
func (t *MemoryPluginStore) GetAll() ([]plugin.LoadedPlugin, error) {
	var allItems []plugin.LoadedPlugin

	for _, item := range t.Store {
		allItems = append(allItems, item)
	}

	return allItems, nil
}
