package nucleus

import (
	"encoding/json"

	"code.fbi.h-da.de/danet/gosdn/controller/customerrs"
	"code.fbi.h-da.de/danet/gosdn/controller/interfaces/networkelement"
	"code.fbi.h-da.de/danet/gosdn/controller/store"
)

// MemoryNetworkElementStore provides a in-memory implementation for network elements.
type MemoryNetworkElementStore struct {
	Store           map[string]networkelement.LoadedNetworkElement
	nameLookupTable map[string]string
}

// NewMemoryNetworkElementStore returns a specific in-memory store for network elements.
func NewMemoryNetworkElementStore() networkelement.Store {
	return &MemoryNetworkElementStore{
		Store:           make(map[string]networkelement.LoadedNetworkElement),
		nameLookupTable: make(map[string]string),
	}
}

// Add adds a item to the store.
func (t *MemoryNetworkElementStore) Add(item networkelement.NetworkElement) error {
	var mne networkelement.LoadedNetworkElement

	b, err := json.Marshal(item)
	if err != nil {
		return err
	}
	err = json.Unmarshal(b, &mne)
	if err != nil {
		return err
	}

	_, ok := t.Store[mne.ID]
	if ok {
		return nil
	}

	t.Store[mne.ID] = mne
	t.nameLookupTable[item.Name()] = mne.ID

	return nil
}

// Update updates a existing network element.
func (t *MemoryNetworkElementStore) Update(item networkelement.NetworkElement) error {
	_, ok := t.Store[item.ID().String()]
	if !ok {
		return customerrs.CouldNotFindError{ID: item.ID(), Name: item.Name()}
	}

	var mne networkelement.LoadedNetworkElement

	b, err := json.Marshal(item)
	if err != nil {
		return err
	}
	err = json.Unmarshal(b, &mne)
	if err != nil {
		return err
	}

	t.Store[item.ID().String()] = mne
	t.nameLookupTable[item.Name()] = item.ID().String()

	return nil
}

// Delete deletes a network element from the network element store.
func (t *MemoryNetworkElementStore) Delete(item networkelement.NetworkElement) error {
	delete(t.Store, item.ID().String())

	return nil
}

// Get takes a network element's UUID or name and returns the network element.
func (t *MemoryNetworkElementStore) Get(query store.Query) (networkelement.LoadedNetworkElement, error) {
	// First search for direct hit on UUID.
	item, ok := t.Store[query.ID.String()]
	if !ok {
		// Second search for name
		id, ok := t.nameLookupTable[query.Name]
		if !ok {
			return item, customerrs.CouldNotFindError{ID: query.ID, Name: query.Name}
		}

		item, ok := t.Store[id]
		if !ok {
			return item, customerrs.CouldNotFindError{ID: query.ID, Name: query.Name}
		}

		return item, nil
	}

	return item, nil
}

// GetAll returns all stored network elements.
func (t *MemoryNetworkElementStore) GetAll() ([]networkelement.LoadedNetworkElement, error) {
	var allItems []networkelement.LoadedNetworkElement

	for _, item := range t.Store {
		allItems = append(allItems, item)
	}

	return allItems, nil
}
