package nucleus

import (
	"fmt"

	"code.fbi.h-da.de/danet/gosdn/controller/interfaces/networkdomain"
	"code.fbi.h-da.de/danet/gosdn/controller/interfaces/networkelement"
	"code.fbi.h-da.de/danet/gosdn/controller/interfaces/plugin"

	"code.fbi.h-da.de/danet/gosdn/controller/customerrs"
	"code.fbi.h-da.de/danet/gosdn/controller/nucleus/database"
	"code.fbi.h-da.de/danet/gosdn/controller/store"
	"github.com/google/uuid"
	log "github.com/sirupsen/logrus"
	"go.mongodb.org/mongo-driver/bson"
	"go.mongodb.org/mongo-driver/bson/primitive"
)

// DatabasePndStore is used to store PrincipalNetworkDomains.
type DatabasePndStore struct {
	pndStoreName    string
	pendingChannels map[uuid.UUID]chan networkelement.Details
	pluginService   plugin.Service
}

// Get takes a PrincipalNetworkDomain's UUID or name and returns the PrincipalNetworkDomain. If the requested
// PrincipalNetworkDomain does not exist an error is returned.
func (s *DatabasePndStore) Get(query store.Query) (newPnd networkdomain.LoadedPnd, err error) {
	var loadedPND networkdomain.LoadedPnd

	if query.ID != uuid.Nil {
		loadedPND, err := s.getByID(query.ID)
		if err != nil {
			return loadedPND, err
		}

		return loadedPND, nil
	}

	loadedPND, err = s.getByName(query.Name)
	if err != nil {
		return loadedPND, err
	}

	// Note: add this if using cSBI again
	// csbiClient, err := s.getCsbiClient()
	// if err != nil {
	// 	return loadedPND, err
	// }

	// newPnd, err := NewPND(
	// 	loadedPND.Name,
	// 	loadedPND.Description,
	// 	uuid.MustParse(loadedPND.ID),
	// )

	return loadedPND, nil
}

func (s *DatabasePndStore) getByID(idOfPnd uuid.UUID) (loadedPnd networkdomain.LoadedPnd, err error) {
	client, ctx, cancel := database.GetMongoConnection()
	defer cancel()
	defer func() {
		if ferr := client.Disconnect(ctx); ferr != nil {
			fErrString := ferr.Error()
			err = fmt.Errorf("InternalError=%w DeferError=%+s", err, fErrString)
		}
	}()

	db := client.Database(database.DatabaseName)
	collection := db.Collection(s.pndStoreName)
	result := collection.FindOne(ctx, bson.D{primitive.E{Key: "_id", Value: idOfPnd.String()}})
	if result == nil {
		return loadedPnd, customerrs.CouldNotFindError{ID: idOfPnd}
	}

	err = result.Decode(&loadedPnd)
	if err != nil {
		log.Printf("Failed marshalling %v", err)
		return loadedPnd, customerrs.CouldNotMarshallError{Identifier: idOfPnd, Type: loadedPnd, Err: err}
	}

	return loadedPnd, nil
}

func (s *DatabasePndStore) getByName(nameOfPnd string) (loadedPnd networkdomain.LoadedPnd, err error) {
	client, ctx, cancel := database.GetMongoConnection()
	defer cancel()
	defer func() {
		if ferr := client.Disconnect(ctx); ferr != nil {
			fErrString := ferr.Error()
			err = fmt.Errorf("InternalError=%w DeferError=%+s", err, fErrString)
		}
	}()

	db := client.Database(database.DatabaseName)
	collection := db.Collection(s.pndStoreName)
	result := collection.FindOne(ctx, bson.D{primitive.E{Key: "name", Value: nameOfPnd}})
	if result == nil {
		return loadedPnd, customerrs.CouldNotFindError{ID: nameOfPnd}
	}

	err = result.Decode(&loadedPnd)
	if err != nil {
		log.Printf("Failed marshalling %v", err)
		return loadedPnd, customerrs.CouldNotMarshallError{Identifier: nameOfPnd, Type: loadedPnd, Err: err}
	}

	return loadedPnd, nil
}

// GetAll returns all stored pnds.
func (s *DatabasePndStore) GetAll() (pnds []networkdomain.LoadedPnd, err error) {
	var loadedPnds []networkdomain.LoadedPnd

	client, ctx, cancel := database.GetMongoConnection()
	defer cancel()
	defer func() {
		if ferr := client.Disconnect(ctx); ferr != nil {
			fErrString := ferr.Error()
			err = fmt.Errorf("InternalError=%w DeferError=%+s", err, fErrString)
		}
	}()

	db := client.Database(database.DatabaseName)
	collection := db.Collection(s.pndStoreName)

	cursor, err := collection.Find(ctx, bson.D{})
	if err != nil {
		return nil, err
	}
	defer func() {
		if ferr := cursor.Close(ctx); ferr != nil {
			fErrString := ferr.Error()
			err = fmt.Errorf("InternalError=%w DeferError=%+s", err, fErrString)
		}
	}()

	err = cursor.All(ctx, &loadedPnds)
	if err != nil {
		log.Printf("Failed marshalling %v", err)

		return nil, customerrs.CouldNotMarshallError{Type: loadedPnds, Err: err}
	}

	// Note: add this if using cSBI again
	// csbiClient, err := s.getCsbiClient()
	// if err != nil {
	// 	return nil, err
	// }

	// for _, loadedPND := range loadedPnds {
	// 	newPnd, err := NewPND(
	// 		loadedPND.Name,
	// 		loadedPND.Description,
	// 		uuid.MustParse(loadedPND.ID),
	// 		csbiClient,
	// 		s.callback,
	// 	)
	// 	if err != nil {
	// 		return nil, err
	// 	}

	// 	pnds = append(pnds, newPnd)
	// }

	return loadedPnds, nil
}

// Add adds a pnd to the pnd store.
func (s *DatabasePndStore) Add(pndToAdd networkdomain.NetworkDomain) (err error) {
	client, ctx, cancel := database.GetMongoConnection()
	defer cancel()
	defer func() {
		if ferr := client.Disconnect(ctx); ferr != nil {
			fErrString := ferr.Error()
			err = fmt.Errorf("InternalError=%w DeferError=%+s", err, fErrString)
		}
	}()

	_, err = client.Database(database.DatabaseName).
		Collection(s.pndStoreName).
		InsertOne(ctx, pndToAdd)
	if err != nil {
		return customerrs.CouldNotCreateError{Identifier: pndToAdd.ID(), Type: pndToAdd, Err: err}
	}

	return nil
}

// Delete deletes a pnd.
// It also deletes all assosicated devices and sbis.
func (s *DatabasePndStore) Delete(pndToDelete networkdomain.NetworkDomain) (err error) {
	client, ctx, cancel := database.GetMongoConnection()
	defer cancel()
	defer func() {
		if ferr := client.Disconnect(ctx); ferr != nil {
			fErrString := ferr.Error()
			err = fmt.Errorf("InternalError=%w DeferError=%+s", err, fErrString)
		}
	}()
	db := client.Database(database.DatabaseName)
	collection := db.Collection(s.pndStoreName)
	_, err = collection.DeleteOne(ctx, bson.D{primitive.E{Key: "_id", Value: pndToDelete.ID().String()}})
	if err != nil {
		return customerrs.CouldNotDeleteError{Identifier: pndToDelete.ID(), Type: pndToDelete, Err: err}
	}

	// TODO: Delete all assosicated devices + SBIs

	return nil
}

// PendingChannels holds channels used communicate with pending
// cSBI deployments.
func (s *DatabasePndStore) PendingChannels(id uuid.UUID, parseErrors ...error) (chan networkelement.Details, error) {
	ch, ok := s.pendingChannels[id]
	if !ok {
		return nil, &customerrs.CouldNotFindError{ID: id}
	}
	return ch, nil
}

// AddPendingChannel adds a pending channel to the map.
func (s *DatabasePndStore) AddPendingChannel(id uuid.UUID, ch chan networkelement.Details) {
	s.pendingChannels[id] = ch
}

// RemovePendingChannel removes a pending channel from the map.
func (s *DatabasePndStore) RemovePendingChannel(id uuid.UUID) {
	delete(s.pendingChannels, id)
}

// func (s *DatabasePndStore) callback(id uuid.UUID, ch chan networkelement.Details) {
// 	if ch != nil {
// 		s.AddPendingChannel(id, ch)
// 		log.Infof("pending channel %v added", id)
// 	} else {
// 		s.RemovePendingChannel(id)
// 		log.Infof("pending channel %v removed", id)
// 	}
// }

// func (s *DatabasePndStore) getCsbiClient() (cpb.CsbiServiceClient, error) {
// 	if s.csbiClient == nil {
// 		orchestrator := viper.GetString("csbi-orchestrator")
// 		conn, err := grpc.Dial(orchestrator, grpc.WithTransportCredentials(insecure.NewCredentials()))
// 		if err != nil {
// 			return nil, err
// 		}

// 		s.csbiClient = cpb.NewCsbiServiceClient(conn)
// 	}

// 	return s.csbiClient, nil
// }
