package app

import (
	"fmt"
	"log"

	"code.fbi.h-da.de/danet/gosdn/controller/customerrs"
	"code.fbi.h-da.de/danet/gosdn/controller/nucleus/database"
	"code.fbi.h-da.de/danet/gosdn/controller/store"

	"github.com/google/uuid"
	"go.mongodb.org/mongo-driver/bson"
	"go.mongodb.org/mongo-driver/bson/primitive"
	"go.mongodb.org/mongo-driver/mongo/options"
)

// ManagementStore is a store for apps.
type ManagementStore interface {
	Add(App) error
	Update(App) error
	Delete(App) error
	Get(store.Query) (App, error)
	GetAll() ([]App, error)
}

// Store stores registered apps.
type Store struct {
	storeName string
}

// NewAppStore returns a AppStore.
func NewAppStore() ManagementStore {
	return &Store{
		storeName: "app-store.json",
	}
}

// Get takes a app's UUID or name and returns the app.
func (s *Store) Get(query store.Query) (App, error) {
	var loadedApp App

	if query.ID.String() != "" && query.ID != uuid.Nil {
		loadedApp, err := s.getByID(query.ID)
		if err != nil {
			return loadedApp, err
		}

		return loadedApp, nil
	}

	loadedApp, err := s.getByName(query.Name)
	if err != nil {
		return loadedApp, err
	}

	return loadedApp, nil
}

func (s *Store) getByID(appID uuid.UUID) (loadedApp App, err error) {
	client, ctx, cancel := database.GetMongoConnection()
	defer cancel()
	defer func() {
		if ferr := client.Disconnect(ctx); ferr != nil {
			fErrString := ferr.Error()
			err = fmt.Errorf("InternalError=%w DeferError=%+s", err, fErrString)
		}
	}()

	db := client.Database(database.DatabaseName)
	collection := db.Collection(s.storeName)
	result := collection.FindOne(ctx, bson.D{primitive.E{Key: "_id", Value: appID.String()}})
	if result == nil {
		return loadedApp, customerrs.CouldNotFindError{ID: appID}
	}

	err = result.Decode(&loadedApp)
	if err != nil {
		log.Printf("Failed marshalling %v", err)
		return loadedApp, customerrs.CouldNotMarshallError{Identifier: appID, Type: loadedApp, Err: err}
	}

	return loadedApp, nil
}

func (s *Store) getByName(appName string) (loadedApp App, err error) {
	client, ctx, cancel := database.GetMongoConnection()
	defer cancel()
	defer func() {
		if ferr := client.Disconnect(ctx); ferr != nil {
			fErrString := ferr.Error()
			err = fmt.Errorf("InternalError=%w DeferError=%+s", err, fErrString)
		}
	}()

	db := client.Database(database.DatabaseName)
	collection := db.Collection(s.storeName)
	result := collection.FindOne(ctx, bson.D{primitive.E{Key: "name", Value: appName}})
	if result == nil {
		return loadedApp, customerrs.CouldNotFindError{Name: appName}
	}

	err = result.Decode(&loadedApp)
	if err != nil {
		log.Printf("Failed marshalling %v", err)
		return loadedApp, customerrs.CouldNotMarshallError{Identifier: appName, Type: loadedApp, Err: err}
	}

	return loadedApp, nil
}

// GetAll returns all stored apps.
func (s *Store) GetAll() (loadedApps []App, err error) {
	client, ctx, cancel := database.GetMongoConnection()
	defer cancel()
	defer func() {
		if ferr := client.Disconnect(ctx); ferr != nil {
			fErrString := ferr.Error()
			err = fmt.Errorf("InternalError=%w DeferError=%+s", err, fErrString)
		}
	}()

	db := client.Database(database.DatabaseName)
	collection := db.Collection(s.storeName)

	cursor, err := collection.Find(ctx, bson.D{})
	if err != nil {
		return nil, err
	}

	err = cursor.All(ctx, &loadedApps)
	if err != nil {
		log.Printf("Failed marshalling %v", err)

		return nil, customerrs.CouldNotMarshallError{Type: loadedApps, Err: err}
	}

	return loadedApps, nil
}

// Add adds a app to the app store.
func (s *Store) Add(app App) (err error) {
	client, ctx, cancel := database.GetMongoConnection()
	defer cancel()
	defer func() {
		if ferr := client.Disconnect(ctx); ferr != nil {
			fErrString := ferr.Error()
			err = fmt.Errorf("InternalError=%w DeferError=%+s", err, fErrString)
		}
	}()

	_, err = client.Database(database.DatabaseName).
		Collection(s.storeName).
		InsertOne(ctx, app)
	if err != nil {
		log.Printf("Could not create app: %v", err)
		return customerrs.CouldNotCreateError{Identifier: app.GetID(), Type: app, Err: err}
	}

	return nil
}

// Update updates a existing app.
func (s *Store) Update(app App) (err error) {
	var updatedApp App

	client, ctx, cancel := database.GetMongoConnection()
	defer cancel()
	defer func() {
		if ferr := client.Disconnect(ctx); ferr != nil {
			fErrString := ferr.Error()
			err = fmt.Errorf("InternalError=%w DeferError=%+s", err, fErrString)
		}
	}()

	update := bson.D{primitive.E{Key: "$set", Value: app}}

	upsert := false
	after := options.After
	opt := options.FindOneAndUpdateOptions{
		Upsert:         &upsert,
		ReturnDocument: &after,
	}

	err = client.Database(database.DatabaseName).
		Collection(s.storeName).
		FindOneAndUpdate(
			ctx, bson.M{"_id": app.GetID().String()}, update, &opt).
		Decode(&updatedApp)
	if err != nil {
		log.Printf("Could not update app: %v", err)

		return customerrs.CouldNotUpdateError{Identifier: app.GetID().String(), Type: app, Err: err}
	}

	return nil
}

// Delete deletes a app from the app store.
func (s *Store) Delete(app App) (err error) {
	client, ctx, cancel := database.GetMongoConnection()
	defer cancel()
	defer func() {
		if ferr := client.Disconnect(ctx); ferr != nil {
			fErrString := ferr.Error()
			err = fmt.Errorf("InternalError=%w DeferError=%+s", err, fErrString)
		}
	}()

	db := client.Database(database.DatabaseName)
	collection := db.Collection(s.storeName)
	_, err = collection.DeleteOne(ctx, bson.D{primitive.E{Key: app.GetID().String()}})
	if err != nil {
		return customerrs.CouldNotDeleteError{Identifier: app.GetID().String(), Type: app, Err: err}
	}

	return nil
}
