package api

import (
	"context"
	"reflect"
	"testing"

	"code.fbi.h-da.de/danet/gosdn/api/go/gosdn/conflict"
	apb "code.fbi.h-da.de/danet/gosdn/api/go/gosdn/rbac"
	"github.com/google/uuid"
	"google.golang.org/grpc/metadata"
)

func TestCreateUsers(t *testing.T) {
	token, err := createTestUserToken("testAdmin", true)
	if err != nil {
		t.Fatalf("%v", err)
	}

	type args struct {
		ctx   context.Context
		addr  string
		users []*apb.User
	}
	tests := []struct {
		name    string
		args    args
		want    apb.Status
		wantErr bool
	}{
		{
			name: "default create users",
			args: args{
				ctx:  metadata.NewOutgoingContext(context.Background(), metadata.Pairs("authorize", token)),
				addr: testAPIEndpoint,
				users: []*apb.User{
					{
						Name:     "foo",
						Roles:    map[string]string{pndID: "s"},
						Password: "roh",
						Token:    "da",
						Metadata: &conflict.Metadata{
							ResourceVersion: 1,
						},
					},
				},
			},
			want:    apb.Status_STATUS_OK,
			wantErr: false,
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			got, err := CreateUsers(tt.args.ctx, tt.args.addr, tt.args.users)
			if (err != nil) != tt.wantErr {
				t.Errorf("CreateUsers() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if !reflect.DeepEqual(got.Status, tt.want) {
				t.Errorf("CreateUsers() = %v, want %v", got, tt.want)
			}
		})
	}
}

func TestGetUser(t *testing.T) {
	type args struct {
		ctx  context.Context
		addr string
		name string
		id   uuid.UUID
	}
	tests := []struct {
		name    string
		args    args
		want    *apb.GetUserResponse
		wantErr bool
	}{
		{
			name: "default get user",
			args: args{
				ctx:  context.TODO(),
				addr: testAPIEndpoint,
				name: "testAdmin",
				id:   uuid.Nil,
			},
			want: &apb.GetUserResponse{
				Status: apb.Status_STATUS_OK,
				User: &apb.User{
					Id:   adminID,
					Name: "testAdmin",
				},
			},
			wantErr: false,
		},
		{
			name: "fail get user",
			args: args{
				ctx:  context.TODO(),
				addr: testAPIEndpoint,
				name: "foos",
				id:   uuid.Nil,
			},
			want:    nil,
			wantErr: true,
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			got, err := GetUser(tt.args.ctx, tt.args.addr, tt.args.name, tt.args.id)
			if (err != nil) != tt.wantErr {
				t.Errorf("GetUser() error = %v, wantErr %v", err, tt.wantErr)
				return
			}

			if got != nil && got.Status == tt.want.Status {
				if got.User.Name != tt.want.User.Name || got.User.Id != tt.want.User.Id {
					t.Errorf("GetUser() = %v, want %v", got, tt.want)
				}
			} else {
				if got != nil {
					t.Errorf("GetUser() = %v, want %v", got, tt.want)
				}
			}
		})
	}
}

func TestGetAllUsers(t *testing.T) {
	err := clearAndCreateAuthTestSetup()
	if err != nil {
		t.Fatalf("%v", err)
	}

	type args struct {
		ctx  context.Context
		addr string
	}
	tests := []struct {
		name    string
		args    args
		want    *apb.GetUsersResponse
		wantLen int
		wantErr bool
	}{
		{
			name: "default get users",
			args: args{
				ctx:  context.TODO(),
				addr: testAPIEndpoint,
			},
			want: &apb.GetUsersResponse{
				Status: apb.Status_STATUS_OK,
				User: []*apb.User{
					{Name: "testAdmin"},
					{Name: "testUser"},
					{Name: "testRandom"}},
			},
			wantLen: 3,
			wantErr: false,
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			got, err := GetAllUsers(tt.args.ctx, tt.args.addr)
			if (err != nil) != tt.wantErr {
				t.Errorf("GetAllUsers() error = %v, wantErr %v", err, tt.wantErr)
				return
			}

			if got != nil && got.Status == apb.Status_STATUS_OK {
				if len(got.User) != tt.wantLen {
					t.Errorf("User.GetUsers() = %v, want %v", got, tt.want)
				}

				for _, gotU := range got.User {
					containsExpected := false
					for _, wantU := range tt.want.User {
						if gotU.Name == wantU.Name {
							containsExpected = true
							break
						}
					}
					if !containsExpected {
						t.Errorf("User.GetUsers() = %v, want %v", got, tt.want)
					}
				}
			}
		})
	}
}

func TestUpdateUsers(t *testing.T) {
	type args struct {
		ctx   context.Context
		addr  string
		users []*apb.User
	}
	tests := []struct {
		name    string
		args    args
		want    *apb.UpdateUsersResponse
		wantErr bool
	}{
		{
			name: "default update user",
			args: args{
				ctx:  context.TODO(),
				addr: testAPIEndpoint,
				users: []*apb.User{
					{
						Id:       adminID,
						Name:     "sth Else",
						Metadata: &conflict.Metadata{},
					},
				},
			},
			want: &apb.UpdateUsersResponse{
				Status: apb.Status_STATUS_OK,
			},
			wantErr: false,
		},
		{
			name: "error update user",
			args: args{
				ctx:  context.TODO(),
				addr: testAPIEndpoint,
				users: []*apb.User{
					{
						Id:   uuid.NewString(),
						Name: "not a User",
					},
				},
			},
			want:    nil,
			wantErr: true,
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			got, err := UpdateUsers(tt.args.ctx, tt.args.addr, tt.args.users)
			if (err != nil) != tt.wantErr {
				t.Errorf("UpdateUsers() error = %v, wantErr %v", err, tt.wantErr)
				return
			}
			if got != nil && got.Status != tt.want.Status {
				t.Errorf("UpdateUsers() = %v, want %v", got, tt.want)
			}
		})
	}
}

func TestDeleteUsers(t *testing.T) {
	type args struct {
		ctx       context.Context
		addr      string
		userNames []string
	}
	tests := []struct {
		name    string
		args    args
		want    *apb.DeleteUsersResponse
		wantErr bool
	}{
		{
			name: "default delete users",
			args: args{
				ctx:       context.TODO(),
				addr:      testAPIEndpoint,
				userNames: []string{"testUser", "testAdmin"},
			},
			want: &apb.DeleteUsersResponse{
				Status: apb.Status_STATUS_OK,
			},
			wantErr: false,
		},
		{
			name: "error delete users",
			args: args{
				ctx:       context.TODO(),
				addr:      testAPIEndpoint,
				userNames: []string{"no User"},
			},
			want:    nil,
			wantErr: true,
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			got, err := DeleteUsers(tt.args.ctx, tt.args.addr, tt.args.userNames)
			if (err != nil) != tt.wantErr {
				t.Errorf("DeleteUsers() error = %v, wantErr %v", err, tt.wantErr)
				return
			}

			if got != nil && got.Status != tt.want.Status {
				t.Errorf("User.DeleteUsers() = %v, want %v", got, tt.want)
			}
		})
	}
}
