/*
Copyright © 2021 da/net Research Group <danet@h-da.de>
All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

3. Neither the name of the copyright holder nor the names of its contributors
   may be used to endorse or promote products derived from this software
   without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.
*/

package cmd

import (
	"errors"
	"fmt"
	"os"

	"code.fbi.h-da.de/danet/gosdn/cli/adapter"

	"github.com/google/uuid"
	log "github.com/sirupsen/logrus"
	"github.com/spf13/cobra"
	"github.com/spf13/viper"
)

var cfgFile string
var verbose bool
var loglevel string
var grpcPort string
var cliPnd string
var controllerAPIEndpoint string
var nbUserName string
var nbUserPwd string
var userToken string

var pndAdapter *adapter.PndAdapter

// rootCmd represents the base command when called without any subcommands.
var rootCmd = &cobra.Command{
	Use:   "gosdnc",
	Short: "goSDN CLI",
	Long: `goSDN CLI using gRPC.
This application can be used to control a goSDN instance.

The init command must be called once per machine.
The login command must be called for authorization.
`,
	Version: "0.9.1",
}

// Execute adds all child commands to the root command and sets flags appropriately.
// This is called by main.main(). It only needs to happen once to the rootCmd.
func Execute() {
	err := rootCmd.Execute()
	if err != nil {
		log.Error("Could not execute root command: ", err)
	}
	//cobra.CheckErr(rootCmd.Execute())
}

func init() {
	cobra.OnInitialize(initConfig)

	// add CLI global parameters
	rootCmd.PersistentFlags().StringVar(&cfgFile, "config", "", "config file (./.gosdnc.toml)")
	rootCmd.PersistentFlags().StringVarP(&loglevel, "log-level", "l", "", "log level 'debug' or 'trace'")
	rootCmd.PersistentFlags().BoolVar(&verbose, "verbose", false, "show mne and sbi info")

	rootCmd.Flags().StringVar(&grpcPort, "grpc-port", "55055", "port for gRPC NBI")
}

// initConfig reads in config file and ENV variables if set.
func initConfig() {
	configFileName := ".gosdnc.toml"
	defaultPath := "./config/"

	if cfgFile != "" {
		// Use config file from the flag.
		viper.SetConfigFile(cfgFile)
	} else {
		// Search config in home directory with name ".gosdnc" (without extension).
		viper.AddConfigPath(defaultPath)
		viper.SetConfigName(configFileName)
		viper.SetConfigType("toml")
	}

	viper.AutomaticEnv() // read in environment variables that match

	// If a config file is found, read it in.
	if err := viper.ReadInConfig(); err != nil {
		if ok := errors.As(err, &viper.ConfigFileNotFoundError{}); ok {
			// create folder if it does not exist
			if err := os.MkdirAll(defaultPath, 0777); err != nil {
				log.Error("Config directory not found and was unable to create, error: ", err)
			}
			// Config file not found
			// Try to create empty config file
			if _, err = os.Create(fmt.Sprintf("%s%s", defaultPath, configFileName)); err != nil {
				log.Error("Config file not found and was unable to create, error: ", err)
			}
		} else {
			// Config file was found but another error was produced
			log.Error("Config file was found but unusable, error:", err)
		}
	}

	cliPnd = viper.GetString("CLI_PND")
	userToken = viper.GetString("USER_TOKEN")

	ll := viper.GetString("GOSDN_LOG")
	if ll != "" {
		loglevel = ll
	}
	log.SetReportCaller(true)
	switch loglevel {
	case "trace":
		log.SetLevel(log.TraceLevel)
	case "debug":
		log.SetLevel(log.DebugLevel)
	default:
		log.SetLevel(log.InfoLevel)
		log.SetReportCaller(false)
	}

	// TODO: Refactor into an general init.
	if cliPnd == "" {
		cliPnd = uuid.New().String()
	}

	adapter, err := adapter.NewPndAdapter(cliPnd, viper.GetString("controllerAPIEndpoint"))
	if err != nil {
		log.Fatal(err)
	}
	pndAdapter = adapter
}
