/*
Copyright © 2021 da/net research group <danet.fbi.h-da.de>
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

3. Neither the name of the copyright holder nor the names of its contributors
   may be used to endorse or promote products derived from this software
   without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.
*/

package cmd

import (
	"fmt"
	"os"

	mnepb "code.fbi.h-da.de/danet/gosdn/api/go/gosdn/networkelement"
	"github.com/google/uuid"
	"github.com/openconfig/gnmi/proto/gnmi"
	"github.com/openconfig/ygot/ygot"
	"github.com/pterm/pterm"
	"github.com/spf13/cobra"
)

var replace bool
var file string
var forcePush bool

// networkElementSetCmd represents the set command.
var networkElementSetCmd = &cobra.Command{
	Use:   "set [uuid] [path] [value]",
	Args:  cobra.RangeArgs(2, 3),
	Short: "set a value on a network element",
	Long: `Set a path value for a given orchestrated network network element. Only one path and
only one value supported for now.

The network element UUID, request path and value must be specified as positional arguments.
To enable replacing behaviour (destructive!), set the --replace flag."`,

	RunE: func(cmd *cobra.Command, args []string) error {
		spinner, _ := pterm.DefaultSpinner.Start("Create a path set request.")
		//NOTE: this currently only works in prompt mode
		if c == nil {
			spinner.Fail("Currently the SET command is only supported with 'gosdnc prompt'")
			return fmt.Errorf("Currently the SET command is only supported with 'gosdnc prompt'")
		}
		mneid, err := uuid.Parse(args[0])
		if err != nil {
			spinner.Fail(err)
			return err
		}
		var value string
		var operation mnepb.ApiOperation
		if replace {
			operation = mnepb.ApiOperation_API_OPERATION_REPLACE
			spinner.UpdateText("Replace generated and sent")
		} else {
			operation = mnepb.ApiOperation_API_OPERATION_UPDATE
			spinner.UpdateText("Update generated and sent")
		}

		completer, ok := c.YangSchemaCompleterMap[mneid]
		if !ok {
			spinner.Fail("No schema completer found for network element ", mneid.String())
			return fmt.Errorf("No schema completer found for network element %s", mneid.String())
		}

		var typedValue *gnmi.TypedValue

		if file == "" && len(args) == 3 {
			if completer.Entry.Type != nil {
				typedValue, err = convertStringToGnmiTypedValue(args[2], completer.Entry.Type)
				if err != nil {
					spinner.Fail(err)
					return err
				}
			} else {
				spinner.Fail("The provided path is no leaf. Please provide a file with JSON-encoded text per RFC7951. Use '--file path' for this.")
				return fmt.Errorf("The provided path is no leaf. Please provide a file with JSON-encoded text per RFC7951. Use '--file path' for this.")
			}

		} else {
			var err error
			value, err = fileContentToString(file)
			if err != nil {
				spinner.Fail(err)
				return err
			}
			typedValue = &gnmi.TypedValue{
				Value: &gnmi.TypedValue_JsonIetfVal{
					JsonIetfVal: []byte(value),
				},
			}
		}

		path, err := ygot.StringToStructuredPath(args[1])
		if err != nil {
			spinner.Fail(err)
			return err
		}

		resp, err := pndAdapter.ChangeMNE(
			createContextWithAuthorization(),
			mneid,
			operation,
			path,
			typedValue,
		)

		if err != nil {
			spinner.Fail(err)
			return err
		}

		for _, r := range resp.GetResponses() {
			spinner.Success("A change for Network element: ", mneid.String(), "has been created -> Change ID: ", r.GetId())
			if forcePush {
				executeFunc("change commit " + r.GetId())
				executeFunc("change confirm " + r.GetId())
			}
		}
		return nil
	},
	PostRun: func(cmd *cobra.Command, args []string) {
		// Necessary for prompt mode. The flag variables have to be resetted,
		// since in prompt mode the program keeps running.
		replace, forcePush, file = false, false, ""
	},
}

func fileContentToString(path string) (string, error) {
	fileContent, err := os.ReadFile(path)
	if err != nil {
		return "", err
	}

	return string(fileContent), nil
}

func init() {
	networkElementCmd.AddCommand(networkElementSetCmd)
	networkElementSetCmd.Flags().BoolVarP(&replace, "replace", "r", false, "enables replace behaviour")
	networkElementSetCmd.Flags().StringVar(&file, "file", "", "reference the path to a file containing your changes as JSON")
	networkElementSetCmd.Flags().BoolVar(&forcePush, "force-push", false, "enables the possibility to instantly push the set without commit/confirm")
}
