/*
Copyright © 2021 da/net research group <danet.fbi.h-da.de>
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

3. Neither the name of the copyright holder nor the names of its contributors
   may be used to endorse or promote products derived from this software
   without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.
*/

package cmd

import (
	"fmt"

	"github.com/google/uuid"
	gpb "github.com/openconfig/gnmi/proto/gnmi"
	"github.com/pterm/pterm"
	"github.com/spf13/cobra"
	"google.golang.org/protobuf/encoding/protojson"
)

// networkElementGetCmd represents the get command.
var networkElementGetIntendedPathCmd = &cobra.Command{
	Use:   "getIntended [uuid] [path]",
	Args:  cobra.ExactArgs(2),
	Short: "gets the intended path from a specified network element on the controller",
	Long: `Requests the intended path from a specified managed network element on the controller.
The network element UUID and request path must be specified as a positional arguments.`,

	RunE: func(cmd *cobra.Command, args []string) error {
		mneid, err := uuid.Parse(args[0])
		if err != nil {
			pterm.Error.Println(err)
			return err
		}

		res, err := pndAdapter.RequestIntendedPath(
			createContextWithAuthorization(),
			mneid,
			args[1],
		)
		if err != nil {
			pterm.Error.Println(err)
			return err
		}

		var stringVal = "Update: "
		var panel3 string
		for _, n := range res.MneNotification {
			panel1 := pterm.DefaultBox.WithTitle("Timestamp:").Sprint(n.GetTimestamp())
			panel2 := pterm.DefaultBox.WithTitle("Requested Path:").Sprint(args[1])

			if len(n.Update) == 0 {
				panel3 = pterm.DefaultBox.WithTitle("Update:").Sprint("empty value")
			} else {
				val := n.Update[0].GetVal()
				switch valTyped := val.GetValue().(type) {
				case *gpb.TypedValue_JsonIetfVal:
					// Note: This is supposed to be a temporary fix to prevent the CLI from blocking during the
					// panel rendering, needs some adjustment in the future.
					stringVal += string(valTyped.JsonIetfVal)
				case *gpb.TypedValue_JsonVal:
					panel3 = pterm.DefaultBox.WithTitle("Update:").Sprint(string(valTyped.JsonVal))
				default:
					panel3 = pterm.DefaultBox.WithTitle("Update:").Sprint(protojson.Format(val))
				}
			}

			panels, _ := pterm.DefaultPanel.WithPanels(pterm.Panels{
				{{Data: panel1}},
				{{Data: panel2}},
				{{Data: panel3}},
			}).Srender()

			pterm.Info.Println(panels)
		}

		if panel3 == "" {
			fmt.Println(stringVal)
		}

		return nil
	},
}

func init() {
	networkElementCmd.AddCommand(networkElementGetIntendedPathCmd)
}
