/*
Copyright © 2021 da/net Research Group <danet@h-da.de>
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

3. Neither the name of the copyright holder nor the names of its contributors
   may be used to endorse or promote products derived from this software
   without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.
*/

package cmd

import (
	"github.com/google/uuid"
	"github.com/pterm/pterm"
	"github.com/spf13/cobra"

	spb "code.fbi.h-da.de/danet/gosdn/api/go/gosdn/southbound"
	tpb "code.fbi.h-da.de/danet/gosdn/api/go/gosdn/transport"
)

// networkElementCreateCmd represents the create command.
var networkElementCreateCmd = &cobra.Command{
	Use:   "create",
	Short: "creates a network element on the controller",
	Long: `Creates an orchestrated network network element on the controller.
Network element address must be provided with IP and port,e.g., 192.168.1.1:6030. User credentials need to be provided as parameters
if they diverge from the default credentials (user:'admin' and pw:'arista').`,

	RunE: func(cmd *cobra.Command, args []string) error {
		spinner, _ := pterm.DefaultSpinner.Start("Creating new network element")
		err := checkIPPort(address)
		if err != nil {
			spinner.Fail(err)
			return err
		}

		opt := &tpb.TransportOption{
			Address:  address,
			Username: username,
			Password: password,
			TransportOption: &tpb.TransportOption_GnmiTransportOption{
				GnmiTransportOption: &tpb.GnmiTransportOption{},
			},
		}
		switch opcode {
		case "plugin":
			opt.Type = spb.Type_TYPE_PLUGIN
		case "csbi":
			opt.Type = spb.Type_TYPE_CONTAINERISED
		default:
			opt.Type = spb.Type_TYPE_OPENCONFIG
		}

		resp, err := pndAdapter.AddNetworkElement(createContextWithAuthorization(), mneName, opt, uuid.MustParse(pluginID))
		if err != nil {
			spinner.Fail(err)
			return err
		}

		for _, r := range resp.GetResponses() {
			spinner.Success("Network element has been created with ID: ", r.GetId())
		}

		return nil
	},
	PostRun: func(cmd *cobra.Command, args []string) {
		// Necessary for prompt mode. The flag variables have to be resetted,
		// since in prompt mode the program keeps running.
		mneName, opcode, pluginID, address, username, password = "", "", "", "", "", ""
	},
}

var mneName string
var opcode string
var pluginID string

func init() {
	networkElementCmd.AddCommand(networkElementCreateCmd)

	networkElementCreateCmd.Flags().StringVar(&mneName, "name", "", "add a network element name (optional)")
	networkElementCreateCmd.Flags().StringVar(&opcode, "type", "", "generation target (csbi or plugin)")
	networkElementCreateCmd.Flags().StringVar(&pluginID, "plugin-id", "", "the plugin ID of the plugin to be used")
	networkElementCreateCmd.Flags().StringVarP(&address, "address", "a", "", "address of a gnmi target, e.g. 192.168.1.1:6030")
	networkElementCreateCmd.Flags().StringVarP(&username, "username", "u", "", "username for a gnmi resource")
	networkElementCreateCmd.Flags().StringVarP(&password, "password", "p", "", "password for a gnmi resource")
}
