/*
Copyright © 2021 da/net Research Group <danet@h-da.de>
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

1. Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

2. Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

3. Neither the name of the copyright holder nor the names of its contributors
   may be used to endorse or promote products derived from this software
   without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
POSSIBILITY OF SUCH DAMAGE.
*/

package cmd

import (
	"context"

	"code.fbi.h-da.de/danet/gosdn/controller/api"
	"github.com/pterm/pterm"
	"github.com/spf13/cobra"
	"github.com/spf13/viper"
)

// loginCmd represents the login command.
var loginCmd = &cobra.Command{
	Use:   "login",
	Short: "Logs in for further actions",
	Long: `Logs the user in to allow further actions on the controller.
    User credentials need to be provided in the body`,

	RunE: func(cmd *cobra.Command, args []string) error {
		spinner, _ := pterm.DefaultSpinner.Start("Login attempt for user: ", nbUserName)

		if controllerAPIEndpoint != "" {
			if err := viper.WriteConfig(); err != nil {
				pterm.Error.Println("Could not write config:", err)
			}
			pterm.Info.Println("New controller address: ", viper.GetString("controllerAPIEndpoint"))
		}

		// log out to remove active session in case an user is already logged in
		if userToken != "" {
			_, _ = api.Logout(createContextWithAuthorization(), viper.GetString("controllerAPIEndpoint"), nbUserName)
		}

		// TODO: maybe add credentials in context instead of context.TODO()
		resp, err := api.Login(context.TODO(), viper.GetString("controllerAPIEndpoint"), nbUserName, nbUserPwd)
		if err != nil {
			spinner.Fail("Login failed: ", err)
			return err
		}

		spinner.Success("Authentication for ", nbUserName, " was successful.")

		userToken = resp.Token

		viper.Set("USER_TOKEN", userToken)

		err = viper.WriteConfig()
		if err != nil {
			pterm.Error.Println(err)
			return err
		}

		pterm.Info.Println("Authentication-User-Token:", userToken)

		return nil
	},
	PostRun: func(cmd *cobra.Command, args []string) {
		// Necessary for prompt mode. The flag variables have to be resetted,
		// since in prompt mode the program keeps running.
		nbUserName, nbUserPwd, controllerAPIEndpoint = "", "", viper.GetString("controllerAPIEndpoint")
	},
}

func init() {
	rootCmd.AddCommand(loginCmd)

	loginCmd.Flags().StringVar(&nbUserName, "u", "", "username for login")
	loginCmd.Flags().StringVar(&nbUserPwd, "p", "", "pwd for login")
	loginCmd.Flags().StringVar(&controllerAPIEndpoint, "controller", viper.GetString("controllerAPIEndpoint"), "address of the controller")

	err := viper.BindPFlag("controllerAPIEndpoint", loginCmd.Flags().Lookup("controller"))
	if err != nil {
		pterm.Error.Println("Could not bind controllerAPIEndpoint:", err)
	}
}
